/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JViewport;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.GridBagUtil;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Advanced JDialog of RepositoryTab
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class AdvancedDialog extends JDialog implements ActionListener {

/*-------------------------------+
| Constants                      |
+-------------------------------*/

	/** Default height */
	protected static final int DEFAULT_HEIGHT = 84;

/*-------------------------------+
| Variables                      |
+-------------------------------*/

	/** Repository id */
	protected String _id;

	/** SystemConfig */
	protected SystemConfig _config;

	/** AccessControlTable */
	protected AccessControlTable _accessControlTable;

	/** SailTable */
	protected SailTable _sailTable;

	/** ParameterTable */
	protected ParameterTable _parameterTable;

	protected JButton _okButton;
	protected JButton _cancelButton;

/*----------------------------------+
| Constructors                      |
+----------------------------------*/
	/**
	 * Creates a new AdvancedDialog with the supplied owner, repository id and
	 * SystemConfig
	 *
	 * @param owner Frame to display this AdvancedDialog in
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public AdvancedDialog(Frame owner, String id, SystemConfig config) {
		// Set title
		super(owner, "Details for repository " + id, true);

		_id = id;
		_config = config;

		Container contentPane = getContentPane();
		contentPane.setLayout(new GridBagLayout());

		// Icon
		//URL url = getClass().getResource("icons/magglass.png");
		URL url = getClass().getResource("icons/details-repository.png");
		JLabel icon = new JLabel(new ImageIcon(url));

		GridBagUtil.constrain(contentPane, icon,
			 0, 0, 1, 1,
			 GridBagConstraints.NONE, GridBagConstraints.CENTER, 0, 0,
			 8, 8, 4, 4);

		// Explanation
		JTextArea explanation = Util.createReadOnlyTextArea(
			"Use the lists below to configure the sail stack and access rights.",
			getBackground());

		GridBagUtil.constrain(contentPane, explanation,
			 1, 0, 1, 1,
			 GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			 8, 4, 0, 8);

		// Tabbed pane
		JTabbedPane tabbedPane = new JTabbedPane();

		GridBagUtil.constrain(contentPane, tabbedPane,
			 0, 1, 2, 1,
			 GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
			 4, 8, 0, 8);

		// Sail stack
		tabbedPane.add("Sail configuration", new SailStackPanel());

		// AccessControlPanel
		tabbedPane.add("Access rights", new AccessControlPanel());

		// Ok/Cancel buttons
		GridBagUtil.constrain(contentPane, _createButtons(),
			0, 2, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.CENTER, 1, 0,
			8, 8, 8, 8);

		pack();

		/*
		if (_sailTable.getRowCount() > 0) {
			// Select first row
			_sailTable.setRowSelectionInterval(0, 0);
		}
		*/

		setLocationRelativeTo(owner);
	}

/*-----------------------------+
| Methods                      |
+-----------------------------*/

	protected JPanel _createButtons() {
		// Use a GridLayout to ensure the buttons get equal sizes:
		JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 4, 0));

		// OK button
		_okButton = new JButton("OK");
		_okButton.setToolTipText("Apply changes and close window");
		buttonPanel.add(_okButton);
		_okButton.addActionListener(this);

		// Cancel button
		_cancelButton = new JButton("Cancel");
		_cancelButton.setToolTipText("Cancel changes and close window");
		buttonPanel.add(_cancelButton);
		_cancelButton.addActionListener(this);

		return buttonPanel;
	}

	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();

		if (source == _okButton) {
			if (_accessControlTable.stopCellEditing() &&
				_sailTable.stopCellEditing() &&
				_parameterTable.stopCellEditing())
			{
				setVisible(false);
				dispose();
			}
			else {
				this.requestFocus();
			}
		}
		else if (source == _cancelButton) {
			setVisible(false);
			dispose();
		}
	}

	/**
	 * Focusses on one of the tables, if they are being edited
	 *
	 * @see javax.swing.JDialog#requestFocus
	 */
	public void requestFocus() {
		if (_accessControlTable.isEditing()) {
			_accessControlTable.requestFocus();
		}
		else if (_sailTable.isEditing()) {
			_sailTable.requestFocus();
		}
		else if (_parameterTable.isEditing()) {
			_parameterTable.requestFocus();
		}
		else {
			super.requestFocus();
		}
	}

/*-----------------------------------+
| Inner class AccessControlPanel     |
+-----------------------------------*/

	/** Access control JPanel of AdvancedDialog */
	protected class AccessControlPanel extends JPanel
		implements ActionListener, ListSelectionListener
	{

	/*----------------------------------+
	| Variables                         |
	+----------------------------------*/

		protected JButton _addButton;
		protected JButton _removeButton;

	/*----------------------------------+
	| Constructors                      |
	+----------------------------------*/
		/** Creates a new AccessControlPanel */
		public AccessControlPanel() {
			setLayout(new GridBagLayout());

			// Add an empty border
			Border emptyBorder = BorderFactory.createEmptyBorder(8, 8, 8, 8);
			this.setBorder(emptyBorder);

			// Title
			JLabel title = new JLabel("Access Control List:");
			GridBagUtil.constrain(this, title,
				0, 0, 1, 1,
				GridBagConstraints.NONE, GridBagConstraints.WEST, 1, 0,
				0, 0, 0, 0);

			// Access Control List
			GridBagUtil.constrain(this, _createList(),
				0, 1, 1, 1,
				GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
				0, 0, 0, 0);

			// Add/Remove buttons
			GridBagUtil.constrain(this, _createButtons(),
				0, 2, 1, 1,
				GridBagConstraints.NONE, GridBagConstraints.EAST, 1, 0,
				4, 0, 0, 0);

			// Add listeners for selections:
			_accessControlTable.getSelectionModel().addListSelectionListener(this);

			// Enable/disable buttons
			_updateButtonStatus();
		}

		protected JScrollPane _createList() {
			JScrollPane list = new JScrollPane();

			_accessControlTable = new AccessControlTable(_id, _config);

			JViewport viewport = list.getViewport();

			viewport.add(_accessControlTable);
			viewport.setBackground(Color.white);

			list.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

			// Set preferred size
			Insets insets = list.getInsets();
			Dimension scrollBarSize = list.getVerticalScrollBar().getPreferredSize();
			Dimension tableSize = _accessControlTable.getPreferredSize();

			int width = insets.left + scrollBarSize.width + tableSize.width + insets.right;

			list.setPreferredSize(new Dimension(width, DEFAULT_HEIGHT));

			return list;
		}

		protected JPanel _createButtons() {
			// Use a GridLayout to ensure the buttons get equal sizes:
			JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 4, 0));

			// Add button
			_addButton = new JButton("Add...");
			_addButton.setToolTipText("Add new user to list");
			buttonPanel.add(_addButton);
			_addButton.addActionListener(this);

			// Remove button
			_removeButton = new JButton("Remove");
			_removeButton.setToolTipText("Remove user from list");
			buttonPanel.add(_removeButton);
			_removeButton.addActionListener(this);

			return buttonPanel;
		}

	/*-----------------------------+
	| Methods                      |
	+-----------------------------*/
		public void actionPerformed(ActionEvent e) {
			Object source = e.getSource();

			if (_accessControlTable.stopCellEditing()) {
				if (source == _addButton) {
					_accessControlTable.addNewRow();
				}
				else if (source == _removeButton) {
					_accessControlTable.removeRow();
				}
			}

			if (_accessControlTable.isEditing()) {
				_accessControlTable.requestFocus();
			}
			else {
				this.requestFocus();
			}
		}

		public void valueChanged(ListSelectionEvent e) {
			_updateButtonStatus();
		}

		/**
		 * Enables/disables buttons according to the current
		 * selection in the Access Control List table.
		 **/
		protected void _updateButtonStatus() {
			int selectedRow = _accessControlTable.getSelectedRow();

			// Disable removeButton when no Sail has been selected (idx == -1),
			// or '-anonymous-' has been selected (idx == 0).
			_removeButton.setEnabled(selectedRow > 0);
		}
	}

/*-----------------------------------+
| Inner class SailStackPanel         |
+-----------------------------------*/

	/** Sail stack JPanel of AdvancedDialog */
	protected class SailStackPanel extends JPanel
		implements ActionListener, ListSelectionListener
	{

	/*----------------------------------+
	| Variables                         |
	+----------------------------------*/

		protected JButton _upButton;
		protected JButton _downButton;
		protected JButton _addButton;
		protected JButton _removeButton;

	/*----------------------------------+
	| Constructors                      |
	+----------------------------------*/
		/** Creates a new SailStackPanel */
		public SailStackPanel() {
			setLayout(new GridBagLayout());

			Border emptyBorder = BorderFactory.createEmptyBorder(8, 8, 8, 8);
			this.setBorder(emptyBorder);

			// Title
			JLabel title = new JLabel("Sail stack:");
			GridBagUtil.constrain(this, title,
				0, 0, 1, 1,
				GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
				0, 0, 0, 0);

			// Sail stack
			GridBagUtil.constrain(this, _createList(),
				0, 1, 1, 1,
				GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
				0, 0, 0, 0);

			// Up/down buttons
			GridBagUtil.constrain(this, _createUpDownButtons(),
				1, 1, 1, 1,
				GridBagConstraints.NONE, GridBagConstraints.EAST, 0, 0,
				0, 4, 0, 0);

			// OK/Cancel buttons
			GridBagUtil.constrain(this, _createOkCancelButtons(),
				0, 2, 1, 1,
				GridBagConstraints.NONE, GridBagConstraints.EAST, 1, 0,
				4, 0, 0, 0);

			// Parameter panel
			ParameterPanel parameterPanel = new ParameterPanel();
			GridBagUtil.constrain(this, parameterPanel,
				0, 3, 1, 1,
				GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
				12, 0, 0, 0);

			// Add listeners for Sail selections:
			_sailTable.getSelectionModel().addListSelectionListener(this);
			_sailTable.getSelectionModel().addListSelectionListener(parameterPanel);

			// Enable/disable buttons
			_updateButtonStatus();
		}

		protected JScrollPane _createList() {
			JScrollPane scrollPane = new JScrollPane();

			_sailTable = new SailTable(_id, _config);

			JViewport viewport = scrollPane.getViewport();

			viewport.add(_sailTable);
			viewport.setBackground(Color.white);

			scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

			// Set preferred size
			Insets insets = scrollPane.getInsets();
			Dimension scrollBarSize = scrollPane.getVerticalScrollBar().getPreferredSize();
			Dimension tableSize = _sailTable.getPreferredSize();

			int width = insets.left + scrollBarSize.width + tableSize.width + insets.right;

			scrollPane.setPreferredSize(new Dimension(width, DEFAULT_HEIGHT));

			return scrollPane;
		}

		protected JPanel _createUpDownButtons() {
			// Use a GridLayout to ensure the buttons get equal sizes:
			JPanel buttonPanel = new JPanel(new GridLayout(2, 1, 0, 4));

			// Up button
			URL upArrowImg = getClass().getResource("icons/uparrow.png");
			_upButton = new JButton(new ImageIcon(upArrowImg));
			_upButton.setToolTipText("Move Sail upwards in stack");
			buttonPanel.add(_upButton);
			_upButton.addActionListener(this);

			// Down button
			URL downArrowImg = getClass().getResource("icons/downarrow.png");
			_downButton = new JButton(new ImageIcon(downArrowImg));
			_downButton.setToolTipText("Move Sail downwards in stack");
			buttonPanel.add(_downButton);
			_downButton.addActionListener(this);

			return buttonPanel;
		}

		protected JPanel _createOkCancelButtons() {
			// Use a GridLayout to ensure the buttons get equal sizes:
			JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 4, 0));

			// Add button
			_addButton = new JButton("Add...");
			_addButton.setToolTipText("Add new Sail to stack");
			buttonPanel.add(_addButton);
			_addButton.addActionListener(this);

			// Remove button
			_removeButton = new JButton("Remove");
			_removeButton.setToolTipText("Remove Sail from stack");
			buttonPanel.add(_removeButton);
			_removeButton.addActionListener(this);

			return buttonPanel;
		}

	/*-----------------------------+
	| Methods                      |
	+-----------------------------*/

		public void actionPerformed(ActionEvent e) {
			Object source = e.getSource();

			if (_sailTable.stopCellEditing()) {
				if (source == _upButton) {
					_sailTable.sailUp();
				}
				else if (source == _downButton) {
					_sailTable.sailDown();
				}
				else if(_parameterTable.stopCellEditing()) {
					if (source == _addButton) {
						_sailTable.addNewRow();
					}
					else if (source == _removeButton) {
						_sailTable.removeRow();
					}
				}
			}

			if (_sailTable.isEditing()) {
				_sailTable.requestFocus();
			}
			else if (_parameterTable.isEditing()) {
				_parameterTable.requestFocus();
			}
		}

		public void valueChanged(ListSelectionEvent e) {
			_updateButtonStatus();
		}

		/**
		 * Enables/disables buttons according to the current
		 * selection in the Sail stack table.
		 **/
		protected void _updateButtonStatus() {
			int selectedRow = _sailTable.getSelectedRow();

			if (selectedRow == -1) {
				// No Sail selected
				_removeButton.setEnabled(false);
				_upButton.setEnabled(false);
				_downButton.setEnabled(false);
			}
			else {
				_removeButton.setEnabled(true);

				// Enable upButton when the selected Sail is
				// not the first Sail.
				_upButton.setEnabled(selectedRow > 0);

				// Enable downButton when the selected Sail
				// is not the last Sail.
				_downButton.setEnabled(selectedRow < _sailTable.getRowCount() - 1);
			}
		}

	/*-----------------------------------+
	| Inner-inner class Parameter Panel  |
	+-----------------------------------*/

		/** Parameter JPanel of SailStackPanel */
		protected class ParameterPanel extends JPanel
			implements ActionListener, ListSelectionListener {

		/*-------------------------------+
		| Variables                      |
		+-------------------------------*/
			/** Class */
			protected String _class;

			protected JScrollPane _scrollPane;

			/** Title */
			protected JLabel _title;

			protected JButton _addButton;
			protected JButton _removeButton;

		/*----------------------------------+
		| Constructors                      |
		+----------------------------------*/

			/** Creates a new ParameterPanel */
			public ParameterPanel() {
				setLayout(new GridBagLayout());

				// Title
				_title = new JLabel("Parameters for selected Sail:");
				GridBagUtil.constrain(this, _title,
					0, 0, 1, 1,
					GridBagConstraints.NONE, GridBagConstraints.WEST, 1, 0,
					0, 0, 0, 0);

				// Parameters
				GridBagUtil.constrain(this, _createList(),
					0, 1, 1, 1,
					GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
					4, 0, 0, 0);

				// Add/Remove buttons
				GridBagUtil.constrain(this, _createButtons(),
					0, 2, 1, 1,
					GridBagConstraints.NONE, GridBagConstraints.EAST, 1, 0,
					4, 0, 0, 0);

				// ParameterPanel is disabled until a Sail is selected
				// in the SailStackPanel
				_enableDisableComponents();
			}

			protected JScrollPane _createList() {
				_parameterTable = new ParameterTable(_id, _config);

				_scrollPane = new JScrollPane();
				JViewport viewport = _scrollPane.getViewport();

				viewport.add(_parameterTable);
				viewport.setBackground(Color.white);

				_scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

				// Set preferred size
				Insets insets = _scrollPane.getInsets();
				Dimension scrollBarSize = _scrollPane.getVerticalScrollBar().getPreferredSize();
				Dimension tableSize = _parameterTable.getPreferredSize();

				int width = insets.left + scrollBarSize.width + tableSize.width + insets.right;

				_scrollPane.setPreferredSize(new Dimension(width, DEFAULT_HEIGHT));

				return _scrollPane;
			}

			protected JPanel _createButtons() {
				// Use a GridLayout to ensure the buttons get equal sizes:
				JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 4, 0));

				// Add button
				_addButton = new JButton("Add...");
				_addButton.setToolTipText("Add new parameter to list");
				buttonPanel.add(_addButton);
				_addButton.addActionListener(this);

				// Remove button
				_removeButton = new JButton("Remove");
				_removeButton.setToolTipText("Remove parameter from list");
				buttonPanel.add(_removeButton);
				_removeButton.addActionListener(this);

				return buttonPanel;
			}

		/*-----------------------------+
		| Methods                      |
		+-----------------------------*/

			/**
			 * Enables/disables components according to which items
			 * are selected.
			 **/
			protected void _enableDisableComponents() {
				int selectedSailIdx = _sailTable.getSelectedRow();

				if (selectedSailIdx == -1) {
					// No Sail selected
					_title.setEnabled(false);

					_scrollPane.setEnabled(false);
					_scrollPane.getViewport().setBackground(getBackground());

					_addButton.setEnabled(false);
					_removeButton.setEnabled(false);
				}
				else {
					// Sail selected
					_title.setEnabled(true);

					_scrollPane.setEnabled(true);
					_scrollPane.getViewport().setBackground(Color.white);

					_addButton.setEnabled(true);

					int selectedParamIdx = _parameterTable.getSelectedRow();
					_removeButton.setEnabled(selectedParamIdx >= 0);
				}
			}

			public void actionPerformed(ActionEvent e) {
				Object source = e.getSource();

				if (_sailTable.stopCellEditing() && _parameterTable.stopCellEditing()) {
					if (source == _addButton) {
						_parameterTable.addNewRow();
					}
					else if (source == _removeButton) {
						_parameterTable.removeRow();
					}
				}

				if (_sailTable.isEditing()) {
					_sailTable.requestFocus();
				}
				else if (_parameterTable.isEditing()) {
					_parameterTable.requestFocus();
				}
			}

			public void valueChanged(ListSelectionEvent e) {
				String sailClass = _sailTable.getIdentifierForSelectedRow();
				_parameterTable.setSailClass(sailClass);

				_enableDisableComponents();
			}
		}
	}
}
