/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.Collections;
import java.util.Iterator;

import javax.swing.JLabel;

import org.openrdf.sesame.config.SystemConfig;

/**
 * Access control XTableModel for AccessControlTable
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.5.4.2 $
 */
public class AccessControlTableModel extends XTableModel {

/*----------+
| Variables |
+----------*/

 	/** Access control login column id */
	public static final int COLUMN_LOGIN = 0;

	/** Access control read column id */
	public static final int COLUMN_READ = 1;

	/** Access control write column id */
	public static final int COLUMN_WRITE = 2;

	/** Anonymous login */
	public static final String ANONYMOUS_LOGIN = "-anonymous-";

	/** Repository id */
	protected String _id;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new AccessControlTableModel with the supplied repository
	 * id and SystemConfig.
	 *
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public AccessControlTableModel(String id, SystemConfig config) {
		super(config, new ColumnData[] {
				new ColumnData("Login", 100, JLabel.LEFT),
				new ColumnData("Read", 25, JLabel.CENTER),
				new ColumnData("Write", 25, JLabel.CENTER)
			});

		_id = id;

		_updateTable();
	}

/*--------+
| Methods |
+--------*/

	/**
	 * -anonymous- login is not editable
	 *
	 * @see org.openrdf.sesame.config.ui.XTableModel#isCellEditable
	 */
	public boolean isCellEditable(int row, int column) {
		if (row == 0 && column == COLUMN_LOGIN) {
			// anonymous login
			return false;
		}
		else {
			return true;
		}
	}

	public int getIdentifyingColumn() {
		return COLUMN_LOGIN;
	}

	public Object getValueAt(int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return null;
		}

		AccessControlData accessControl = (AccessControlData)_rows.get(row);

		switch (column) {
			case COLUMN_LOGIN:
				return accessControl.getLogin();

			case COLUMN_READ:
				return Boolean.valueOf(accessControl.hasReadAccess());

			case COLUMN_WRITE:
				return Boolean.valueOf(accessControl.hasWriteAccess());
		}

		return null;
	}

	public void setValueAt(Object value, int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return;
		}

		AccessControlData accessControl = (AccessControlData)_rows.get(row);

 		switch(column) {
			case COLUMN_LOGIN:
				String stringValue = (String)value;

				if (accessControl.isNew()) {
					accessControl.setLogin(stringValue);

					_config.setReadAccess(_id, accessControl.getLogin(), true);
				}
				else {
					String login = accessControl.getLogin();

					if (!login.equals(stringValue)) {
						// Login has changed
						boolean readAccess = _config.hasReadAccess(_id, login);
						boolean writeAccess = _config.hasWriteAccess(_id, login);

						_config.setReadAccess(_id, login, false);
						_config.setWriteAccess(_id, login, false);

						_config.setReadAccess(_id, login, readAccess);
						_config.setWriteAccess(_id, login, writeAccess);
					}
				}

				break;

			case COLUMN_READ:
				boolean readAccess = ((Boolean)value).booleanValue();

				if (row == 0) {
					_config.setWorldReadable(_id, readAccess);
				}
				else {
					_config.setReadAccess(_id, accessControl.getLogin(), readAccess);
				}

				break;

			case COLUMN_WRITE:
				boolean writeAccess = ((Boolean)value).booleanValue();

				if (row == 0) {
					_config.setWorldWriteable(_id, writeAccess);
				}
				else {
					_config.setWriteAccess(_id, accessControl.getLogin(), writeAccess);
				}

				break;
		}
	}

	public void configurationChanged() {
		_updateTable();
	}

	protected void _updateTable() {
		Iterator rowsIter = _rows.iterator();

		while (rowsIter.hasNext()) {
			AccessControlData accessControl = (AccessControlData)rowsIter.next();

			if (accessControl.isNew()) {
				// Table is being edtited, do not update
				return;
			}
		}

		_rows.clear();

		if (!_config.hasRepository(_id)) {
			// Repository does not exist anymore
			_id = null;
		}
		else {
			// Create default anonymous account which represents world
			// readability and writeability of the repository
			_rows.add(new AccessControlData(
					ANONYMOUS_LOGIN, _config.isWorldReadable(_id), _config.isWorldWriteable(_id)));

			// Add users that have read and/or write access.
			Iterator loginIter = _config.getUsernames().iterator();

			while (loginIter.hasNext()) {
				String login = (String)loginIter.next();

				if (_config.hasReadOrWriteAccess(_id, login)) {
					boolean readAccess = _config.hasReadAccess(_id, login);
					boolean writeAccess = _config.hasWriteAccess(_id, login);

					_rows.add(new AccessControlData(login, readAccess, writeAccess));
				}
			}

			// Sort the rows.
			Collections.sort(_rows);
		}

		fireTableDataChanged();
	}

	protected RowData _createRow() {
		return new AccessControlData();
	}

/*------------------------------+
| Inner class AccessControlData |
+------------------------------*/

	/**
	 * Access control data of AccessControlTableModel
	 **/
	class AccessControlData extends RowData {

		/*----------------------+
		| Variables             |
		+----------------------*/

		/** User login */
		protected String _login;

		/** Indicates if this user has read access */
		protected boolean _readAccess;

		/** Indicates if this user has write access */
		protected boolean _writeAccess;

		/*----------------------+
		| Constructors          |
		+----------------------*/

		/**
		 * Creates a new AccessControlData
		 **/
		public AccessControlData() {
		}

		/**
		 * Creates a new AccessControlData with the supplied user login and if
		 * this user has read and/or write access
		 *
		 * @param login User login
		 * @param readAccess Boolean indicating if this user has read access
		 * @param writeAccess Boolean indicating if this user has write access
		 */
		public AccessControlData(String login, boolean readAccess, boolean writeAccess) {
			_login = login;
			_readAccess = readAccess;
			_writeAccess = writeAccess;
		}

		/*----------------------+
		| Methods               |
		+----------------------*/

		public String getIdentifier() {
			return getLogin();
		}

		/**
		 * Gets the user login
		 *
		 * @return User login
		 */
		public String getLogin() {
			return _login;
		}

		/**
		 * Sets the user login to the supplied login
		 *
		 * @param User login
		 */
		public void setLogin(String login) {
			_login = login;
		}

		/**
		 * Checks if this user has read access
		 *
		 * @return Boolean indicating if this user has read access
		 */
		public boolean hasReadAccess() {
			return _readAccess;
		}

		/**
		 * Checks if this user has write access
		 *
		 * @return Boolean indicating if this user has write access
		 */
		public boolean hasWriteAccess() {
			return _writeAccess;
		}

		public boolean isNew() {
			return _login == null;
		}

		public int compareTo(Object o) {
			AccessControlData other = (AccessControlData)o;

			if (_login == null) {
				return 1;
			}
			else if (other._login == null) {
				return -1;
			}
			else if (_login.equals(ANONYMOUS_LOGIN)) {
				return -1;
			}
			else if (other._login.equals(ANONYMOUS_LOGIN)) {
				return 1;
			}
			else {
				return _login.compareTo(other._login);
			}
		}
	}
}
