/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.awt.Component;
import java.util.Arrays;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Access control XTable of RepositoryTab
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.5.4.2 $
 */
public class AccessControlTable extends XTable {

/*----------+
| Variables |
+----------*/

	/** Repository id */
	protected String _repositoryId;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new AccessControlTable with the supplied repository id and
	 * SystemConfig
	 *
	 * @param repositoryId Repository id
	 * @param config SystemConfig
	 */
	public AccessControlTable(String repositoryId, SystemConfig config) {
		super(config);
	    _repositoryId = repositoryId;

		setXTableModel(new AccessControlTableModel(repositoryId, config));

		// Select first row
		setRowSelectionInterval(0, 0);
	}

/*--------+
| Methods |
+--------*/

	protected TableCellRenderer _createCellRenderer(int columnNo) {
		if (columnNo == AccessControlTableModel.COLUMN_LOGIN) {
			DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}
		else if (columnNo == AccessControlTableModel.COLUMN_READ ||
				 columnNo == AccessControlTableModel.COLUMN_WRITE)
		{
			CheckCellRenderer renderer = new CheckCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}

		return null;
	}

	protected TableCellEditor _createCellEditor(int columnNo) {
		TableCellEditor editor = null;

		if (columnNo == AccessControlTableModel.COLUMN_LOGIN) {
			// Create alphabetically sorted login names list
			Object[] items = _config.getUsernames().toArray();
			Arrays.sort(items);
			editor = new LoginCellEditor(items, this);
		}
		else if (columnNo == AccessControlTableModel.COLUMN_READ ||
				 columnNo == AccessControlTableModel.COLUMN_WRITE)
		{
			JCheckBox editorCheckBox = new JCheckBox();

			editor = new XCellEditor(editorCheckBox, this);

			editorCheckBox.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
		}

		return editor;
	}

	/**
	 * Adds a new row after anonymous login
	 *
	 * @see org.openrdf.sesame.config.ui.XTable#addNewRow
	 */
	public void addNewRow() {
		if (!_config.hasAnUser()) {
			_showWarningDialog("No users configured yet.", "Add User");
			return;
		}

		int row = getSelectedRow();

		if (row == -1) {
			// No row selected, add a new row at index 1
			row = 1;
		}
		else {
			row++;
		}

		_model.addNewRow(row);
		editCellAt(row, AccessControlTableModel.COLUMN_LOGIN);
	}

	public void removeRow() {
		String login = getIdentifierForSelectedRow();

		if (login == null) {
			_showWarningDialog("No user selected.", "Remove User");
			return;
		}

		if (login.equals(AccessControlTableModel.ANONYMOUS_LOGIN)) {
			_showWarningDialog(
					AccessControlTableModel.ANONYMOUS_LOGIN +
					" cannot be removed", "Remove User");
			return;
		}

		if (Util.showYesNoDialog(_getOwner(),
			 	"Are you sure you want to remove user '" + login +
				"' from the Access Control List? ",
			 	"Remove User") == JOptionPane.YES_OPTION)
		{
			int row = getSelectedRow();
			_config.setReadAccess(_repositoryId, login, false);
			_config.setWriteAccess(_repositoryId, login, false);
			selectPreviousRowTo(row);
		}
	}

/*-----------------------------------------+
| Inner class LoginCellEditor              |
+-----------------------------------------*/

	class LoginCellEditor extends XCellEditor {
	
	/*------------------+
	| Variables         |
	+------------------*/

		/** Indicates if the value was just set */
		protected boolean valueWasSet;
	
	/*------------------+
	| Constructors      |
	+------------------*/

		public LoginCellEditor(Object[] items, AccessControlTable acTable) {
			super(new JComboBox(items), acTable);
		}
	
	/*------------------+
	| Methods           |
	+------------------*/

		/**
		 * The value was just set
		 *
		 * @see org.openrdf.sesame.config.ui.XCellEditor#setValue
		 */
		public void setValue(Object value) {
			super.setValue(value);
			valueWasSet = true;
		}

		/**
		 * For some reason, Java calls stopCellEditing when a
		 * cell is being edited and the JComboBox is displayed
		 * for the first time. Therefore, checks if the value was
		 * just set. If so, does not validate input.
		 *
		 * @see org.openrdf.sesame.config.ui.XCellEditor#stopCellEditing
		 */
		public boolean stopCellEditing() {
			if (valueWasSet) {
				valueWasSet = false;
				return false;
			}
			else {
				return super.stopCellEditing();
			}
		}

		public boolean isValid() {
			String oldLogin = (String)_value;
			String newLogin = getCellEditorValue().toString();

			if (!newLogin.equals(oldLogin) &&
				_config.hasReadOrWriteAccess(_repositoryId, newLogin))
			{
				// Login has changed
				_showWarningDialog(
						"Access for user '" + newLogin + "' already defined.",
						"Edit Access");
				return false;
			}

			return true;
		}

		/**
		 * Selects item at index 0 of combo box
		 *
		 * @see org.openrdf.sesame.config.ui.XCellEditor#getTableCellEditorComponent
		 */
		public Component getTableCellEditorComponent(
			JTable table, Object value,
			boolean isSelected, int row, int column)
		{
			if (value == null) {
				JComboBox box = (JComboBox)getComponent();
				value = box.getItemAt(0);
			}

			return super.getTableCellEditorComponent(
					table, value, isSelected, row, column);
		}
	}
}
