/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.handlers;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.openrdf.util.xml.XmlWriter;

import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.UserInfo;

class SystemConfigWriter {

/*-------------------------------------------+
| Variables                                  |
+-------------------------------------------*/

	protected SystemConfig _config;
	protected XmlWriter _xmlWriter;

/*-------------------------------------------+
| Constructors                               |
+-------------------------------------------*/

	public SystemConfigWriter() {
	}

/*-------------------------------------------+
| Methods                                    |
+-------------------------------------------*/

	public synchronized void write(SystemConfig systemConfig, OutputStream outputStream)
		throws IOException
	{
		_write(systemConfig, new XmlWriter(outputStream));
	}

	public synchronized void write(SystemConfig systemConfig, Writer writer)
		throws IOException
	{
		_write(systemConfig, new XmlWriter(writer));
	}

	private void _write(SystemConfig systemConfig, XmlWriter xmlWriter)
		throws IOException
	{
		_config = systemConfig;
		_xmlWriter = xmlWriter;
		_xmlWriter.setPrettyPrint(true);

		_xmlWriter.startDocument();

		_writeHeader();

		_xmlWriter.emptyLine();
		_xmlWriter.comment("server parameters");
		_writeAdmin();
		_writeLog();
		_writeTmp();
		_writeRmiFactory();

		_xmlWriter.emptyLine();
		_xmlWriter.comment("users");
		_writeUsers();

		_xmlWriter.emptyLine();
		_xmlWriter.comment("repositories");
		_writeRepositories();

		_writeFooter();

		_xmlWriter.endDocument();
	}

	protected void _writeHeader()
		throws IOException
	{
		_xmlWriter.startTag("system-conf");
	}

	protected void _writeFooter()
		throws IOException
	{
		_xmlWriter.endTag("system-conf");
	}

	protected void _writeAdmin()
		throws IOException
	{
		_xmlWriter.setAttribute("password", _config.getAdminPassword());
		_xmlWriter.emptyElement("admin");
	}

	protected void _writeLog()
		throws IOException
	{
		String logDir = _config.getLogDir();
		if (logDir == null) {
			logDir = "";
		}
		_xmlWriter.setAttribute("dir", logDir);
		_xmlWriter.setAttribute("level", _config.getLogLevel());
		_xmlWriter.emptyElement("log");
	}

	protected void _writeTmp()
		throws IOException
	{
		String tmpDir = _config.getTmpDir();
		if (tmpDir == null) {
			tmpDir = "";
		}
		_xmlWriter.setAttribute("dir", tmpDir);
		_xmlWriter.emptyElement("tmp");
	}

	protected void _writeRmiFactory()
		throws IOException
	{
		if (_config.getRMIFactoryClass() != null) {
			_xmlWriter.setAttribute("enabled", _config.isRMIEnabled());
			_xmlWriter.setAttribute("class", _config.getRMIFactoryClass());
			_xmlWriter.setAttribute("port", _config.getRMIPort());
			_xmlWriter.emptyElement("rmi-factory");
		}
	}

	protected void _writeUsers()
		throws IOException
	{
		_xmlWriter.startTag("userlist");

		List userInfoList = _config.getUserInfoList();
		for (int i = 0; i < userInfoList.size(); i++) {
			UserInfo userInfo = (UserInfo)userInfoList.get(i);

			if (i > 0) {
				_xmlWriter.emptyLine();
			}

			_xmlWriter.setAttribute("id", userInfo.getID());
			_xmlWriter.setAttribute("login", userInfo.getLogin());
			_xmlWriter.startTag("user");

			_xmlWriter.textElement("fullname", userInfo.getFullName());
			_xmlWriter.textElement("password", userInfo.getPassword());

			_xmlWriter.endTag("user");
		}

		_xmlWriter.endTag("userlist");
	}

	protected void _writeRepositories()
		throws IOException
	{
		_xmlWriter.startTag("repositorylist");

		List repConfigList = _config.getRepositoryConfigList();
		for (int i = 0; i < repConfigList.size(); i++) {
			RepositoryConfig repConfig = (RepositoryConfig)repConfigList.get(i);

			if (i > 0) {
				_xmlWriter.emptyLine();
			}

			_xmlWriter.setAttribute("id", repConfig.getRepositoryId());
			_xmlWriter.startTag("repository");

			_xmlWriter.textElement("title", repConfig.getTitle());

			_writeSailStack(repConfig);
			_writeAcl(repConfig);

			_xmlWriter.endTag("repository");
		}

		_xmlWriter.endTag("repositorylist");
	}

	protected void _writeSailStack(RepositoryConfig repConfig)
		throws IOException
	{
		_xmlWriter.startTag("sailstack");

		List sailList = repConfig.getSailList();
		for (int i = 0; i < sailList.size(); i++) {
			SailConfig sailConfig = (SailConfig)sailList.get(i);
			_writeSail(sailConfig);
		}

		_xmlWriter.endTag("sailstack");
	}

	protected void _writeSail(SailConfig sailConfig)
		throws IOException
	{
		Map configParams = sailConfig.getConfigParameters();

		_xmlWriter.setAttribute("class", sailConfig.getSailClass());

		if (configParams.isEmpty()) {
			// Sail without configuration parameters
			_xmlWriter.emptyElement("sail");
		}
		else {
			// Sail with configuration parameters
			_xmlWriter.startTag("sail");

			Iterator paramNames = configParams.keySet().iterator();
			while (paramNames.hasNext()) {
				String name = (String)paramNames.next();
				String value = (String)configParams.get(name);

				_xmlWriter.setAttribute("name", name);
				_xmlWriter.setAttribute("value", value);
				_xmlWriter.emptyElement("param");
			}

			_xmlWriter.endTag("sail");
		}
	}

	protected void _writeAcl(RepositoryConfig repConfig)
		throws IOException
	{
		List userInfoList = _config.getUsersForRepository(repConfig);

		_xmlWriter.setAttribute("worldReadable", repConfig.isWorldReadable());
		_xmlWriter.setAttribute("worldWriteable", repConfig.isWorldWriteable());

		if (userInfoList.isEmpty()) {
			// Without users
			_xmlWriter.emptyElement("acl");
		}
		else {
			// With users
			_xmlWriter.startTag("acl");

			String repositoryId = repConfig.getRepositoryId();

			for (int i = 0; i < userInfoList.size(); i++) {
				UserInfo user = (UserInfo)userInfoList.get(i);

				_xmlWriter.setAttribute("login", user.getLogin());
				_xmlWriter.setAttribute("readAccess", user.hasReadAccess(repositoryId));
				_xmlWriter.setAttribute("writeAccess", user.hasWriteAccess(repositoryId));
				_xmlWriter.emptyElement("user");
			}

			_xmlWriter.endTag("acl");
		}
	}
}
