/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.handlers;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.log.ThreadLog;
import org.openrdf.util.xml.SimpleSAXAdapter;
import org.openrdf.util.xml.SimpleSAXParser;

import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.UserInfo;

class SystemConfigReader {

/*-----------------------+
| Variables              |
+-----------------------*/

	private SimpleSAXParser _simpleSAXParser;

/*-----------------------+
| Constructors           |
+-----------------------*/

	/**
	 * Creates a new SystemConfigReader that will use the supplied
	 * <tt>XMLReader</tt> for parsing the XML-based system
	 * configuration documents.
	 **/
	public SystemConfigReader(XMLReader xmlReader) {
		_simpleSAXParser = new SimpleSAXParser(xmlReader);
	}

/*-----------------------+
| Methods                |
+-----------------------*/

	public synchronized SystemConfig read(InputStream inputStream)
		throws SAXException, IOException
	{
		SystemConfigParser parser = new SystemConfigParser();
		_simpleSAXParser.setListener(parser);
		_simpleSAXParser.parse(inputStream);
		return parser.getSystemConfig();
	}

	public synchronized SystemConfig read(Reader reader)
		throws SAXException, IOException
	{
		SystemConfigParser parser = new SystemConfigParser();
		_simpleSAXParser.setListener(parser);
		_simpleSAXParser.parse(reader);
		return parser.getSystemConfig();
	}

/*-------------------------------+
| Inner class SystemConfigParser |
+-------------------------------*/

	static class SystemConfigParser extends SimpleSAXAdapter {

	/*----------------+
	| Variables       |
	+----------------*/

		private SystemConfig _config;

		private UserInfo _currentUser;
		private RepositoryConfig _currentRepository;
		private SailConfig _currentSail;

		private boolean _inAcl;

	/*----------------+
	| Methods         |
	+----------------*/

		public SystemConfig getSystemConfig() {
			return _config;
		}

		public void startDocument() {
			// Initialize variables
			_currentUser = null;
			_currentRepository = null;
			_currentSail = null;
			_inAcl = false;
		}

		public void endDocument()
			throws SAXException
		{
			// Check if admin password was set
			if (_config.getAdminPassword() == null) {
				ThreadLog.warning("admin password was not set!!!");
			}
		}
	
		public void startTag(String tagName, Map atts, String text) {
			if (tagName.equals("system-conf")) {
				_config = new SystemConfig();
			}
			else if (tagName.equals("admin")) {
				String password = (String)atts.get("password");
				if (password == null) {
					ThreadLog.error("'password' attribute missing in 'admin' element");
					return;
				}
				_config.setAdminPassword(password);
			}
			else if (tagName.equals("log")) {
				String dir = (String)atts.get("dir");
				if (dir == null) {
					ThreadLog.error("'dir' attribute missing in 'log' element");
					return;
				}
				_config.setLogDir(dir);
	
				int logLevel = 2;
				String logLevelStr = (String)atts.get("level");
				if (logLevelStr != null) {
					try {
						logLevel = Integer.parseInt(logLevelStr);
						if (logLevel >= 0 && logLevel  <= 5) {
							_config.setLogLevel(logLevel);
						}
						else {
							ThreadLog.error("value of 'level' attribute in 'log' element should be a number between 0 and 5");
						}
					}
					catch (NumberFormatException e) {
						ThreadLog.error("value of 'level' attribute in 'log' element should be a number between 0 and 5");
					}
				}
			}
			else if (tagName.equals("tmp")) {
				String dir = (String)atts.get("dir");
				if (dir == null) {
					ThreadLog.error("'dir' attribute missing in 'tmp' element");
					return;
				}
				_config.setTmpDir(dir);
			}
			else if (tagName.equals("rmi-factory")) {
				String facClass = (String)atts.get("class");
				if (facClass == null || facClass.trim().length() == 0) {
					ThreadLog.error("'class' attribute missing in 'rmi-factory' element");
				}
	
				int portInt = java.rmi.registry.Registry.REGISTRY_PORT;
				String port = (String)atts.get("port");
				if (port != null) {
					try {
						portInt = Integer.parseInt(port);
					}
					catch (NumberFormatException e) {
						ThreadLog.error("value of 'port' attribute in 'rmi-factory' element should be a positive number");
					}
				}
				_config.setRMIFactory(facClass, portInt);
	
				String rmiEnabled = (String)atts.get("enabled");
				if (rmiEnabled == null) {
					ThreadLog.warning("'enabled' attribute missing in 'rmi-factory' element");
				}
	
				_config.setRMIEnabled(rmiEnabled == null || rmiEnabled.equals("true"));
			}
			else if (tagName.equals("systemprop")) {
				String key = (String)atts.get("key");
				if (key == null) {
					ThreadLog.error("'key' attribute missing in 'systemprop' element");
					return;
				}
	
				String value = (String)atts.get("value");
				if (value == null) {
					ThreadLog.error("'value' attribute missing in 'systemprop' element");
					return;
				}
	
				_config.setSystemProp(key, value);
			}
			else if (tagName.equals("user") && !_inAcl) {
				_currentUser = null;
				_currentRepository = null;
	
				String login = (String)atts.get("login");
				if (login == null) {
					ThreadLog.error("'login' attribute missing in 'user' element");
					return;
				}
	
				_currentUser = new UserInfo(login.trim());
	
				String userID = (String)atts.get("id");
				if (userID != null) {
					try {
						_currentUser.setID(Integer.parseInt(userID.trim()));
					}
					catch (NumberFormatException e) {
						ThreadLog.error("value of 'id' attribute in 'user' element should be a positive number");
					}
				}
			}
			else if (tagName.equals("fullname")) {
				_currentUser.setFullName(text.trim());
			}
			else if (tagName.equals("password")) {
				_currentUser.setPassword(text.trim());
			}
			else if (tagName.equals("repository")) {
				String id = (String)atts.get("id");
				if (id == null) {
					ThreadLog.error("'id' attribute missing in 'repository' element");
					return;
				}
				_currentRepository = new RepositoryConfig(id);
				_currentUser = null;
			}
			else if (tagName.equals("title")) {
				_currentRepository.setTitle(text.trim());
			}
			else if (tagName.equals("sail")) {
				if (_currentRepository != null) {
					String sailClass = (String)atts.get("class");
					if (sailClass == null) {
						ThreadLog.error("'class' attribute missing in 'sail' element");
						return;
					}
	
					_currentSail = new SailConfig(sailClass);
					_currentRepository.addSail(_currentSail);
				}
			}
			else if (tagName.equals("param")) {
				if (_currentSail != null) {
					String paramName = (String)atts.get("name");
					String paramValue = (String)atts.get("value");
					if (paramName == null) {
						ThreadLog.error("'name' attribute missing in 'param' element");
						return;
					}
					if (paramValue == null) {
						ThreadLog.error("'value' attribute missing in 'param' element");
						return;
					}
	
					_currentSail.setParameter(paramName, paramValue);
				}
			}
			else if (tagName.equals("acl")) {
				if (_currentRepository != null) {
					_inAcl = true;
					String worldReadable = (String)atts.get("worldReadable");
					String worldWriteable = (String)atts.get("worldWriteable");
	
					if (worldWriteable == null) {
						// FIXME: old config files use wrongly spelled version,
						// this was changed in 0.97.
						worldWriteable = (String)atts.get("worldWritable");
					}
	
					_currentRepository.setWorldReadable(
							worldReadable != null && worldReadable.equals("true"));
	
					_currentRepository.setWorldWriteable(
							worldWriteable != null && worldWriteable.equals("true"));
	
				}
			}
			else if (tagName.equals("user") && _inAcl) {
				String login = (String)atts.get("login");
				String readAccess = (String)atts.get("readAccess");
				String writeAccess = (String)atts.get("writeAccess");
				if (login == null) {
					ThreadLog.error("'login' attribute missing in 'user' element for acl");
					return;
				}
				UserInfo userInfo = _config.getUserInfo(login);
				if (userInfo == null) {
					ThreadLog.error("user '" + login + "' not defined");
					return;
				}
	
				if (readAccess != null && readAccess.equals("true")) {
					userInfo.addReadAccess(_currentRepository);
				}
	
				if (writeAccess != null && writeAccess.equals("true")) {
					userInfo.addWriteAccess(_currentRepository);
				}
			}
		}
	
		public void endTag(String tagName) {
			if (tagName.equals("user") && !_inAcl) {
				if (_currentUser != null) {
					_addCurrentUser();
					_currentUser = null;
				}
			}
			else if (tagName.equals("repository")) {
				if (_currentRepository != null) {
					_addCurrentRepository();
					_currentRepository = null;
					_inAcl = false;
				}
			}
			else if (tagName.equals("acl")) {
				_inAcl = false;
			}
		}
	
		private void _addCurrentUser() {
			boolean correctConfig = true;
	
			String login = _currentUser.getLogin();
	
			// Check if all parameters are set.
			if (_currentUser.getFullName() == null) {
				ThreadLog.error("fullname missing for user " + login);
				correctConfig = false;
			}
			if (_currentUser.getPassword() == null) {
				ThreadLog.error("password missing for user " + login);
				correctConfig = false;
			}
			if (_currentUser.getID() == 0) {
				ThreadLog.warning("id attribute missing for user " + login +
						"; using default");
			}
	
			if (correctConfig) {
				ThreadLog.trace("user " + login + " was parsed correctly");
				_config.addUserInfo(_currentUser);
			}
			else {
				ThreadLog.warning("ignoring user " + login);
			}
		}
	
		private void _addCurrentRepository() {
			boolean correctConfig = true;
	
			String id = _currentRepository.getRepositoryId();
	
			// Check if all parameters are set.
			if (_currentRepository.getTitle() == null) {
				ThreadLog.error("title missing for repository " + id);
				correctConfig = false;
			}
			if (!_currentRepository.hasASail()) {
				ThreadLog.error("no sail defined for repository " + id);
				correctConfig = false;
			}
	
			if (correctConfig) {
				ThreadLog.trace("repository " + id + " was parsed correctly");
				_config.addRepositoryConfig(_currentRepository);
			}
			else {
				ThreadLog.warning("ignoring repository " + id);
			}
		}
	}
}
