/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.handlers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.log.ThreadLog;
import org.openrdf.util.xml.XMLReaderFactory;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.SystemConfigHandler;
import org.openrdf.sesame.server.SesameServer;

public class SystemConfigFileHandler implements SystemConfigHandler {

/*-----------------------------------+
| Variables                          |
+-----------------------------------*/

	private static final String SYSCONFIGFILE = "systemConfigFile";

	protected Map _parameters;

/*-----------------------------------+
| Methods                            |
+-----------------------------------*/

	public void setParameters(Map parameters) {
		_parameters = parameters;
	}

	public Set getParameterNames() {
		Set result = new HashSet();
		result.add(SYSCONFIGFILE);
		return result;
	}

	public SystemConfig loadConfig()
		throws IOException
	{
		String sysConfigFile = (String)_parameters.get(SYSCONFIGFILE);

		// Resolve possible relative path:
		File f = new File(SesameServer.getBaseDir(), sysConfigFile);

		try {
			InputStream in = new FileInputStream(f);
			SystemConfig newConfig = readConfiguration(in);
			in.close();
			return newConfig;
		}
		catch (FileNotFoundException e) {
			throw e;
		}
		catch (IOException e) {
			throw e;
		}
	}

	public void storeConfig(SystemConfig config)
		throws IOException
	{
		String sysConfigFile = (String)_parameters.get(SYSCONFIGFILE);

		// Resolve possible relative path:
		File f = new File(SesameServer.getBaseDir(), sysConfigFile);

		try {
			OutputStream out = new FileOutputStream(f);
			writeConfiguration(config, out);
			out.close();
		}
		catch (IOException e) {
			ThreadLog.error("unable to store system configuration file '" + f + "'", e);
			throw e;
		}
	}

	/**
	 * Reads a system configuration from the supplied <tt>InputStream</tt>.
	 * Before calling this method, one should set the system property
	 * <tt>org.xml.sax.driver</tt> to the name of a class that implements
	 * the interface <tt>org.xml.sax.XMLReader</tt>. This method does not
	 * close the supplied InputStream, this is the responsibility of the
	 * caller.
	 **/
	public static SystemConfig readConfiguration(InputStream inputStream)
		throws IOException
	{
		// Parse the configuration file
		try {
			SystemConfigReader scr = _createSystemConfigReader();
			SystemConfig systemConfig = scr.read(inputStream);
			return systemConfig;
		}
		catch (SAXException e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * Reads a system configuration from the supplied <tt>Reader</tt>.
	 * Before calling this method, one should set the system property
	 * <tt>org.xml.sax.driver</tt> to the name of a class that implements
	 * the interface <tt>org.xml.sax.XMLReader</tt>. This method does not
	 * close the supplied Reader, this is the responsibility of the caller.
	 **/
	public static SystemConfig readConfiguration(Reader reader)
		throws IOException
	{
		// Parse the configuration file
		try {
			SystemConfigReader scr = _createSystemConfigReader();
			SystemConfig systemConfig = scr.read(reader);
			return systemConfig;
		}
		catch (SAXException e) {
			throw new IOException(e.getMessage());
		}
	}

	private static SystemConfigReader _createSystemConfigReader() {
		// Try to create an XMLReader
		XMLReader xmlReader;
		try {
			xmlReader = XMLReaderFactory.createXMLReader();
		}
		catch (Exception e) {
			ThreadLog.error("Unable to create XMLReader", e);
			throw new RuntimeException(e.getMessage());
		}

		return new SystemConfigReader(xmlReader);
	}

	/**
	 * Writes the specified system configuration to the supplied
	 * <tt>OutputStream</tt>. This method does not close the supplied
	 * OutputStream, this is the responsibility of the caller.
	 **/
	public static void writeConfiguration(SystemConfig config, OutputStream outputStream)
		throws IOException
	{
		SystemConfigWriter sysConfWriter = new SystemConfigWriter();
		sysConfWriter.write(config, outputStream);
		outputStream.flush();
	}

	/**
	 * Writes the specified system configuration to the supplied
	 * <tt>Writer</tt>. This method does not close the supplied
	 * Writer, this is the responsibility of the caller.
	 **/
	public static void writeConfiguration(SystemConfig config, Writer writer)
		throws IOException
	{
		SystemConfigWriter sysConfWriter = new SystemConfigWriter();
		sysConfWriter.write(config, writer);
		writer.flush();
	}
}
