/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config;

import java.util.HashMap;
import java.util.Map;

public class UserInfo implements Cloneable {

/*--------------------+
| Variables           |
+--------------------*/

	private int _id;
	private String _login;
	private String _fullName;
	private String _password;

	/** A List of RepositoryConfig objects. **/
	private Map _readAccessMap;

	/** A List of RepositoryConfig objects. **/
	private Map _writeAccessMap;

/*--------------------+
| Constructors        |
+--------------------*/

	public UserInfo(String login) {
		_login = login;
		_readAccessMap = new HashMap();
		_writeAccessMap = new HashMap();
	}

	/**
	 * Creates a new UserInfo with the supplied user id, login, full name and
	 * password
	 *
	 * @param id User id
	 * @param login User login
	 * @param fullName User full name
	 * @param password User password
	 */
	public UserInfo(int id, String login, String fullName, String password) {
		this(login);

		_id = id;
		_fullName = fullName;
		_password = password;
	}

/*--------------------+
| Methods             |
+--------------------*/

	public int getID() {
		return _id;
	}

	public void setID(int id) {
		_id = id;
	}

	public String getLogin() {
		return _login;
	}

	public void setLogin(String login) {
		_login = login;
	}

	public String getFullName() {
		return _fullName;
	}

	public void setFullName(String fullName) {
		_fullName = fullName;
	}

	public String getPassword() {
		return _password;
	}

	public void setPassword(String password) {
		_password = password;
	}

	public boolean hasReadAccess(String repositoryId) {
		return _readAccessMap.containsKey(repositoryId);
	}

	public void addReadAccess(RepositoryConfig repConfig) {
		_readAccessMap.put(repConfig.getRepositoryId(), repConfig);
	}

	public void removeReadAccess(RepositoryConfig repConfig) {
		_readAccessMap.remove(repConfig.getRepositoryId());
	}

	public boolean hasWriteAccess(String repositoryId) {
		return _writeAccessMap.containsKey(repositoryId);
	}

	public void addWriteAccess(RepositoryConfig repConfig) {
		_writeAccessMap.put(repConfig.getRepositoryId(), repConfig);
	}

	public void removeWriteAccess(RepositoryConfig repConfig) {
		_writeAccessMap.remove(repConfig.getRepositoryId());
	}

	public void addReadWriteAccess(RepositoryConfig repConfig) {
		addReadAccess(repConfig);
		addWriteAccess(repConfig);
	}

	public boolean hasReadOrWriteAccess(String repositoryId) {
		return hasReadAccess(repositoryId) || hasWriteAccess(repositoryId);
	}

	public boolean equals(Object other) {
		if (other instanceof UserInfo) {
			UserInfo o = (UserInfo)other;

			return
				_login.equals(o.getLogin()) &&
				_fullName.equals(o.getFullName()) &&
				_password.equals(o.getPassword()) &&
				_id == o.getID();
		}

		return false;
	}

	public int hashCode() {
		return _login.hashCode();
	}

	// Overrides Object.clone()
	public Object clone() {
		try {
			UserInfo clone = (UserInfo)super.clone();

			// Create copies of the read- and write access maps
			clone._readAccessMap = new HashMap(_readAccessMap);
			clone._writeAccessMap = new HashMap(_writeAccessMap);

			return clone;
		}
		catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
	}
}
