/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2007 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

public class SystemConfig {

/*------------------------------------------+
| Variables                                 |
+------------------------------------------*/

	protected String _adminPassword = "";

	protected String _logDir;
	protected int _logLevel = 0;

	protected String _tmpDir;

	protected boolean _rmiEnabled;
	protected String _rmiFactoryClass;
	protected int _rmiPort;

	protected Properties _systemProps = new Properties();

	protected HashMap _userMap = new HashMap();
	protected List _userList = new ArrayList();

	protected HashMap _repositoryMap = new HashMap();
	protected ArrayList _repositoryList = new ArrayList();

	protected List _listeners = new ArrayList();

/*------------------------------------------+
| Constructors                              |
+------------------------------------------*/

	public SystemConfig() {
	}

/*------------------------------------------+
| Server-related methods                    |
+------------------------------------------*/

	public void setAdminPassword(String password) {
		_adminPassword = password;
		_notifyListeners();
	}

	public String getAdminPassword() {
		return _adminPassword;
	}

	public void setLogDir(String logDir) {
		_logDir = logDir;
		_notifyListeners();
	}

	public String getLogDir() {
		return _logDir;
	}

	public void setLogLevel(int logLevel) {
		_logLevel = logLevel;
		_notifyListeners();
	}

	public int getLogLevel() {
		return _logLevel;
	}

	public void setTmpDir(String tmpDir) {
		_tmpDir = tmpDir;
		_notifyListeners();
	}

	public String getTmpDir() {
		return _tmpDir;
	}

	public void setSystemProp(String key, String value) {
		_systemProps.setProperty(key, value);
		_notifyListeners();
	}

	public String getSystemProp(String key) {
		return _systemProps.getProperty(key);
	}

	public Properties getSystemProps() {
		return _systemProps;
	}

	public void setRMIEnabled(boolean rmiEnabled) {
		if (_rmiEnabled != rmiEnabled) {
			_rmiEnabled = rmiEnabled;
			_notifyListeners();
		}
	}

	public boolean isRMIEnabled() {
		return _rmiEnabled;
	}

	public void setRMIFactory(String rmiFactoryClass, int port) {
		_rmiFactoryClass = rmiFactoryClass;
		_rmiPort = port;
		_notifyListeners();
	}

	public String getRMIFactoryClass() {
		return _rmiFactoryClass;
	}

	public int getRMIPort() {
		return _rmiPort;
	}

/*------------------------------------------+
| User-related methods                      |
+------------------------------------------*/

	public void addUserInfo(UserInfo ui) {
		_userMap.put(ui.getLogin(), ui);
		_userList.add(ui);
		_notifyListeners();
	}

	/**
	 * Removes the user with the supplied login
	 *
	 * @param login User login
	 */
	public void removeUser(String login) {
		UserInfo user = (UserInfo)_userMap.remove(login);
		_userList.remove(user);
		_notifyListeners();
	}

	public UserInfo getUserInfo(String login) {
		return (UserInfo)_userMap.get(login);
	}

	public UserInfo getUserInfo(int userID) {
		for (int i = 0; i < _userList.size(); i++) {
			UserInfo ui = (UserInfo)_userList.get(i);

			if (ui.getID() == userID) {
				return ui;
			}
		}

		return null;
	}

	public Set getUsernames() {
		return Collections.unmodifiableSet(_userMap.keySet());
	}

	/**
	 * Gets a List of users
	 *
	 * @return a List of UserInfo objects.
	 */
	public List getUserInfoList() {
		return Collections.unmodifiableList(_userList);
	}

	/**
	 * Gets all users that have read and/or write access to the
	 * specified repository.
	 **/
	public List getUsersForRepository(RepositoryConfig repConfig) {
		List result = new ArrayList();

		String repositoryId = repConfig.getRepositoryId();

		for (int i = 0; i < _userList.size(); i++) {
			UserInfo userInfo = (UserInfo)_userList.get(i);
			if (userInfo.hasReadOrWriteAccess(repositoryId)) {
				result.add(userInfo);
			}
		}

		return result;
	}

	/**
	 * Gets an unused user id.
	 *
	 * @return An unused user id.
	 */
	public int getUnusedUserId() {
		// Collect used ids
		int[] usedIDs = new int[_userList.size()];
		for (int i = 0; i < _userList.size(); i++) {
			UserInfo userInfo = (UserInfo)_userList.get(i);
			usedIDs[i] = userInfo.getID();
		}

		Arrays.sort(usedIDs);

		// User id 's 1 and 2 are reserved for the ADMIN and ANONYMOUS accounts
		// respectively. Begin at id 3.
		int nextUserId = 3;

		// find an unused user id
		while (Arrays.binarySearch(usedIDs, nextUserId) >= 0) {
			nextUserId++;
		}

		return nextUserId;
	}

	/**
	 * Sets the id of the user with the supplied login to the given id
	 *
	 * @param login User login
	 * @param id User id
	 */
	public void setUserId(String login, int id) {
		getUserInfo(login).setID(id);
		_notifyListeners();
	}

	/**
	 * Sets the login of the user with the supplied old login to the given new
	 * login
	 *
	 * @param oldLogin Old user login
	 * @param newLogin New user login
	 */
	public void setUserLogin(String oldLogin, String newLogin) {
		UserInfo userInfo = getUserInfo(oldLogin);

		userInfo.setLogin(newLogin);

		_userMap.remove(oldLogin);
		_userMap.put(newLogin, userInfo);

		_notifyListeners();
	}

	/**
	 * Sets the full name of the user with the supplied login to the given full
	 * name
	 *
	 * @param login User login
	 * @param fullName User full name
	 */
	public void setUserFullName(String login, String fullName) {
		getUserInfo(login).setFullName(fullName);
		_notifyListeners();
	}

	/**
	 * Sets the password of the user with the supplied login to the given
	 * password
	 *
	 * @param login User login
	 * @param password User password
	 */
	public void setUserPassword(String login, String password) {
		getUserInfo(login).setPassword(password);
		_notifyListeners();
	}

/*------------------------------------------+
| Repository-related methods                |
+------------------------------------------*/

	public void addRepositoryConfig(RepositoryConfig ri) {
		RepositoryConfig oldInfo = (RepositoryConfig)_repositoryMap.put(ri.getRepositoryId(), ri);
		if (oldInfo != null) {
			_repositoryList.remove(oldInfo);
		}
		_repositoryList.add(ri);

		_notifyListeners();
	}

	public RepositoryConfig getRepositoryConfig(String id) {
		return (RepositoryConfig)_repositoryMap.get(id);
	}

	public List getRepositoryConfigList() {
		return Collections.unmodifiableList(_repositoryList);
	}

	/**
	 * Gets a List of sails from the repository with the supplied id
	 *
	 * @param id Repository id
	 * @return an unmodifiable List of SailConfig objects.
	 * @see SailConfig
	 */
	public List getSails(String id) {
		return getRepositoryConfig(id).getSailList();
	}

	/**
	 * Gets a Map of parameters from the sail with the supplied class of the
	 * repository with the given id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * return Map of parameters
	 */
	public Map getParameters(String id, String sailClass) {
		return getRepositoryConfig(id).getSail(sailClass).getConfigParameters();
	}

	/**
	 * Sets the id of the repository with the supplied old id to the given new id
	 *
	 * @param oldId Old repository id
	 * @param newId New repository id
	 */
	public void setRepositoryId(String oldId, String newId) {
		RepositoryConfig repository = getRepositoryConfig(oldId);

		repository.setRepositoryId(newId);

		_repositoryMap.remove(oldId);
		_repositoryMap.put(newId, repository);

		_notifyListeners();
	}

	/**
	 * Sets the title of the repository with the supplied id to the given
	 * title
	 *
	 * @param id Repository id
	 * @param title Repository title
	 */
	public void setRepositoryTitle(String id, String title) {
		getRepositoryConfig(id).setTitle(title);
		_notifyListeners();
	}

	/**
	 * Checks if the repository with the supplied id is world readable
	 *
	 * @param id Repository id
	 * @return Boolean indicating if the repository is world readable
	 */
	public boolean isWorldReadable(String id) {
		return getRepositoryConfig(id).isWorldReadable();
	}

	/**
	 * Sets if the repository with the supplied id is world readable
	 *
	 * @param id Repository id
	 * @param worldReadable Boolean indicating if the repository is world
	 * readable
	 */
	public void setWorldReadable(String id, boolean worldReadable) {
		getRepositoryConfig(id).setWorldReadable(worldReadable);
		_notifyListeners();
	}

	/**
	 * Checks if the repository with the supplied id is world writeable
	 *
	 * @param id The repository id
	 * @return Boolean indicating if the repository is world writeable
	 */
	public boolean isWorldWriteable(String id) {
		return getRepositoryConfig(id).isWorldWriteable();
	}

	/**
	 * Sets if the repository with the supplied id is world writeable
	 *
	 * @param id The repository id
	 * @param worldWriteable Boolean indicating if the repository is world
	 * writeable
	 */
	public void setWorldWriteable(String id, boolean worldWriteable) {
		getRepositoryConfig(id).setWorldWriteable(worldWriteable);
		_notifyListeners();
	}

	/**
	 * Sets the sail with the supplied old class of the repository with the
	 * given id to the supplied new class
	 *
	 * @param id Repository id
	 * @param oldClass Old sail class
	 * @param newClass New sail class
	 */
	public void setSail(String id, String oldClass, String newClass) {
		getRepositoryConfig(id).getSail(oldClass).setSailClass(newClass);
		_notifyListeners();
	}

	/**
	 * Sets the key of the parameter with the supplied old key of the sail with
	 * the given class of the repository with the supplied id to the given new
	 * key
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param oldKey Old parameter key
	 * @param newKey New parameter key
	 */
	public void setParameterKey(String id, String sailClass, String oldKey, String newKey) {
		getRepositoryConfig(id).getSail(sailClass).setParameterKey(oldKey, newKey);
		_notifyListeners();
	}

	/**
	 * Sets the value of the parameter with the supplied key of the sail with the
	 * given class of the repository with the supplied id to the given value
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param key Parameter key
	 * @param value Parameter value
	 */
	public void setParameterValue(String id, String sailClass, String key, String value) {
		getRepositoryConfig(id).getSail(sailClass).setParameter(key, value);

		_notifyListeners();
	}

	/**
	 * Checks if an user exists
	 *
	 * @return Boolean indicating if an user exists
	 */
	public boolean hasAnUser() {
		return !_userList.isEmpty();
	}

	/**
	 * Checks if an user with the supplied login exists
	 *
	 * @param login User login
	 * @return Boolean indicating if the user exists
	 */
	public boolean hasUser(String login) {
		return _userMap.containsKey(login);
	}

	/**
	 * Checks if a repository exists
	 *
	 * @return Boolean indicating if a repository exists
	 */
	public boolean hasARepository() {
		return !_repositoryList.isEmpty();
	}

	/**
	 * Checks if a repository with the supplied id exists
	 *
	 * @param id The repository id
	 * @return Boolean indicating if the repository exists
	 */
	public boolean hasRepository(String id) {
		return _repositoryMap.containsKey(id);
	}

	/**
	 * Checks if the repository with the supplied id has a sail
	 *
	 * @param id Repository id
	 * @return Boolean indicating if the repository has a sail
	 */
	public boolean hasASail(String id) {
		return getRepositoryConfig(id).hasASail();
	}

	/**
	 * Checks if the repository with the supplied id has a sail with the supplied
	 * class
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @return Boolean indicating if the repository has the sail
	 */
	public boolean hasSail(String id, String sailClass) {
		return getRepositoryConfig(id).hasSail(sailClass);
	}

	/**
	 * Checks if the sail with the supplied class of the repository with the
	 * given id has a parameter with the supplied key
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param key Parameter key
	 * @return Boolean indicating if the sail of the repository has the parameter
	 */
	public boolean hasParameter(String id, String sailClass, String key) {
		return getRepositoryConfig(id).getSail(sailClass).hasParameter(key);
	}

	/**
	 * Adds a new user with the supplied id, login, full name and password
	 *
	 * @param id User id
	 * @param login User login
	 * @param fullName User full name
	 * @param password User password
	 */
	public void addUser(int id, String login, String fullName, String password) {
		addUserInfo(new UserInfo(id, login, fullName, password));
	}

	/**
	 * Adds a new repository with the supplied id and title
	 *
	 * @param id Repository id
	 * @param title Repository title
	 */
	public void addRepository(String id, String title) {
		addRepositoryConfig(new RepositoryConfig(id, title));
	}

	/**
	 * Clones the repository with the specified id, changes its id to
	 * the specified new id, and adds this new repository.
	 *
	 * @param origID The repository to clone.
	 * @param cloneID The ID of the cloned repository.
	 */
	public RepositoryConfig cloneRepository(String origID, String cloneID) {
		RepositoryConfig repConfig = getRepositoryConfig(origID);

		repConfig = (RepositoryConfig)repConfig.clone();
		repConfig.setRepositoryId(cloneID);

		addRepositoryConfig(repConfig);

		return repConfig;
	}

	/**
	 * Adds the sail with the supplied class to the sail stack of the repository
	 * with the given id at the supplied index
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param idx Sail index
	 */
	public void addSail(String id, String sailClass, int idx) {
		getRepositoryConfig(id).addSail(new SailConfig(sailClass), idx);
		_notifyListeners();
	}

	/**
	 * Adds the parameter with the supplied key-value pair to the sail with the
	 * given class of the repository with the supplied id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param key Parameter key
	 * @param value Parameter value
	 */
	public void setParameter(String id, String sailClass, String key, String value) {
		getRepositoryConfig(id).getSail(sailClass).setParameter(key, value);
		_notifyListeners();
	}

	/**
	 * Removes the repository with the supplied id
	 *
	 * @param id Repository id
	 */
	public void removeRepository(String id) {
		RepositoryConfig repository = (RepositoryConfig)_repositoryMap.remove(id);
		_repositoryList.remove(repository);
		_notifyListeners();
	}

	/**
	 * Removes the sail with the supplied class from the sail stack of the
	 * repository with the supplied id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 */
	public void removeSail(String id, String sailClass) {
		RepositoryConfig repository = getRepositoryConfig(id);
		repository.removeSail(repository.getSail(sailClass));
		_notifyListeners();
	}

	/**
	 * Removes the parameter with the supplied key from the sail with the given
	 * class of the repository with the supplied id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 * @param key Parameter key
	 */
	public void removeParameter(String id, String sailClass, String key) {
		getRepositoryConfig(id).getSail(sailClass).removeParameter(key);
		_notifyListeners();
	}

	/**
	 * Moves the sail with the supplied class one level up in the sail stack of
	 * the repository with the given id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 */
	public void sailUp(String id, String sailClass) {
		getRepositoryConfig(id).sailUp(sailClass);
		_notifyListeners();
	}

	/**
	 * Moves the sail with the supplied class one level down in the sail stack of
	 * the repository with the given id
	 *
	 * @param id Repository id
	 * @param sailClass Sail class
	 */
	public void sailDown(String id, String sailClass) {
		getRepositoryConfig(id).sailDown(sailClass);
		_notifyListeners();
	}

	/**
	 * Checks if the user with the supplied login has read access to
	 * the specified repository.
	 *
	 * @param id Repository id
	 * @param login User login
	 * @return Boolean indicating if the user has read access to the repository
	 */
	public boolean hasReadAccess(String id, String login) {
		boolean result = false;

		RepositoryConfig repository = getRepositoryConfig(id);
		if (repository.isWorldReadable()) {
			result = true;
		}
		else if (login != null) {
			UserInfo user = getUserInfo(login);
			if (user != null) {
				result = user.hasReadAccess(id);
			}
		}

		return result;
	}

	/**
	 * Sets the read access of the user with supplied login to the repository
	 * with the given id to the supplied read access
	 *
	 * @param id Repository id
	 * @param login User login
	 * @param readAccess Boolean indicating if the user has read access to the
	 * repository
	 */
	public void setReadAccess(String id, String login, boolean readAccess) {
		UserInfo user = getUserInfo(login);
		RepositoryConfig repository = getRepositoryConfig(id);

		if (readAccess == true) {
			user.addReadAccess(repository);
		}
		else {
			user.removeReadAccess(repository);
		}

		_notifyListeners();
	}

	/**
	 * Checks if the user with the supplied login has write access to
	 * the specified repository.
	 *
	 * @param id Repository id
	 * @param login User login
	 * @return Boolean indicating if the user has write access to the repository
	 */
	public boolean hasWriteAccess(String id, String login) {
		boolean result = false;

		RepositoryConfig repository = getRepositoryConfig(id);
		if (repository.isWorldWriteable()) {
			result = true;
		}
		else if (login != null) {
			UserInfo user = getUserInfo(login);
			if (user != null) {
				result = user.hasWriteAccess(id);
			}
		}

		return result;
	}

	/**
	 * Sets the write access of the user with supplied login to the repository
	 * with the given id to the supplied write access
	 *
	 * @param id Repository id
	 * @param login User login
	 * @param writeAccess Boolean indicating if the user has write access to the
	 * repository
	 */
	public void setWriteAccess(String id, String login, boolean writeAccess) {
		UserInfo user = getUserInfo(login);
		RepositoryConfig repository = getRepositoryConfig(id);

		if (writeAccess == true) {
			user.addWriteAccess(repository);
		}
		else {
			user.removeWriteAccess(repository);
		}

		_notifyListeners();
	}

	/**
	 * Checks if the user with the supplied login has read or write
	 * access to the specified repository.
	 *
	 * @param id Repository id
	 * @param login User login
	 * @return Boolean indicating if the user has access to the repository
	 */
	public boolean hasReadOrWriteAccess(String id, String login) {
		return hasReadAccess(id, login) || hasWriteAccess(id, login);
	}

	/**
	 * Checks if the user with the supplied login has both read and
	 * write access to the specified repository.
	 *
	 * @param id Repository id
	 * @param login User login
	 * @return Boolean indicating if the user has access to the repository
	 */
	public boolean hasReadAndWriteAccess(String id, String login) {
		return hasReadAccess(id, login) && hasWriteAccess(id, login);
	}

	/**
	 * Copies the data from <tt>other</tt> to this SystemConfig.
	 *
	 * @param other The SystemConfig to copy.
	 */
	public void setSystemConfig(SystemConfig other) {
		_adminPassword = other._adminPassword;
		_logDir = other._logDir;
		_logLevel = other._logLevel;
		_tmpDir = other._tmpDir;
		_rmiEnabled = other._rmiEnabled;
		_rmiFactoryClass = other._rmiFactoryClass;
		_rmiPort = other._rmiPort;
		_systemProps = new Properties(other._systemProps);
		_userMap = new HashMap(other._userMap);
		_userList = new ArrayList(other._userList);
		_repositoryMap = new HashMap(other._repositoryMap);
		_repositoryList = new ArrayList(other._repositoryList);

		_notifyListeners();
	}

	/**
	 * Adds the supplied SystemConfigListener
	 *
	 * @param listener SystemConfigListener
	 */
	public void addListener(SystemConfigListener listener) {
		_listeners.add(listener);
	}

	/**
	 * Removes the supplied SystemConfigListener
	 *
	 * @param listener SystemConfigListener
	 */
	public void removeListener(SystemConfigListener listener) {
		_listeners.remove(listener);
	}

	protected void _notifyListeners() {
		Iterator i = _listeners.iterator();

		while (i.hasNext()) {
			SystemConfigListener listener = (SystemConfigListener)i.next();

			listener.configurationChanged();
		}
	}
}
