/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config;

import java.util.HashMap;
import java.util.Map;

/**
 * Representation of the configuration settings for a Sail object.
 */
public class SailConfig implements Cloneable {

	private String _sailClass;
	private Map _configParams = new HashMap();

	public SailConfig(String sailClass) {
		_sailClass = sailClass;
	}

	/**
	 * Returns the sail class of the Sail object represented by this
	 * SailConfig.
	 */
	public String getSailClass() {
		return _sailClass;
	}

	/**
	 * Sets the sail class of the Sail object represented by this
	 * SailConfig.
	 *
	 * @param sailClass the full class name of the Sail object.
	 */
	public void setSailClass(String sailClass) {
		_sailClass = sailClass;
	}

	/**
	 * Sets the parameter with the supplied value.
	 *
	 * @param name the name of the parameter
	 * @param value the value of the parameter
	 */
	public void setParameter(String name, String value) {
		_configParams.put(name, value);
	}

	/**
	 * Removes the parameter with the supplied key
	 *
	 * @param key Parameter key
	 */
	public void removeParameter(String key) {
		_configParams.remove(key);
	}

	/**
	 * Retrieves the parameter with the supplied name.
	 *
	 * @param name the name of the parameter
	 *
	 * @return the value of the parameter if set, null otherwise.
	 */
	public String getParameter(String name) {
		return (String)_configParams.get(name);
	}

	/**
	 * Sets the key of the parameter with the supplied old key to the given new
	 * key
	 *
	 * @param oldKey Old parameter key
	 * @param newKey New parameter key
	 */
	public void setParameterKey(String oldKey, String newKey) {
		String value = getParameter(oldKey);

		removeParameter(oldKey);
		setParameter(newKey, value);
	}

	/**
	 * Sets the value of the parameter with the supplied key to the given value
	 *
	 * @param key Parameter key
	 * @param value Parameter value
	 */
	public void setParameterValue(String key, String value) {
		_configParams.put(key, value);
	}

	/**
	 * Checks if the parameter with the supplied key exists
	 *
	 * @param key Parameter key
	 * @return Boolean indicating if the parameter exists
	 */
	public boolean hasParameter(String key) {
		return _configParams.containsKey(key);
	}

	public Map getConfigParameters() {
		return _configParams;
	}

	public void setConfigParameters(Map configParams) {
		_configParams = configParams;
	}

	// Overrides Object.equals(Object)
	public boolean equals(Object other) {
		if (other instanceof SailConfig) {
			SailConfig o = (SailConfig)other;

			return
				_sailClass.equals(o.getSailClass()) &&
				_configParams.equals(o.getConfigParameters());
		}

		return false;
	}

	// Overrides Object.hashCode()
	public int hashCode() {
		return _sailClass.hashCode();
	}

	// Overrides Object.clone()
	public Object clone() {
		try {
			SailConfig clone = (SailConfig)super.clone();

			// Create a copy of the config parameters map
			clone.setConfigParameters(new HashMap(_configParams));

			return clone;
		}
		catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
	}
}
