/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.xml.SimpleSAXAdapter;
import org.openrdf.util.xml.SimpleSAXParser;

/**
 * Parser for parsing repository lists returned by Sesame. The parser
 * assumes that the format of the acquired data is correct, and
 * doesn't do any extra verification.
 **/
public class RepositoryListReader {

/*--------------------+
| Variables           |
+--------------------*/

	/**
	 * The SimpleSAXParser that is used for parsing the XML.
	 **/
	private SimpleSAXParser _simpleSAXParser;

/*--------------------+
| Constructors        |
+--------------------*/

	public RepositoryListReader(XMLReader xmlReader) {
		_simpleSAXParser = new SimpleSAXParser(xmlReader);
	}

/*--------------------+
| Methods             |
+--------------------*/

	public synchronized void read(InputStream in, RepositoryListListener list)
		throws SAXException, IOException
	{
		_simpleSAXParser.setListener(new XmlRepositoryListReader(list));
		_simpleSAXParser.parse(in);
	}

/*------------------------------------+
| Inner class XmlRepositoryListReader |
+------------------------------------*/

	static class XmlRepositoryListReader extends SimpleSAXAdapter {

		/**
		 * The listener to report the repositories to.
		 **/
		private RepositoryListListener _listener;

		private String _currentReposId;
		private boolean _currentReposReadable;
		private boolean _currentReposWritable;
		private String _currentReposTitle;

		public XmlRepositoryListReader(RepositoryListListener list) {
			_listener = list;
		}

		public void startTag(String tagName, Map atts, String text) {
			if (tagName.equals("repositorylist")) {
				_listener.startRepositoryList();
			}
			else if (tagName.equals("repository")) {
				_currentReposId = (String)atts.get("id");
				_currentReposReadable = "true".equals( atts.get("readable") );
				_currentReposWritable = "true".equals( atts.get("writeable") );
			}
			else if (tagName.equals("title")) {
				_currentReposTitle = text;
			}
		}

		public void endTag(String tagName) {
			if (tagName.equals("repositorylist")) {
				_listener.endRepositoryList();
			}
			else if (tagName.equals("repository")) {
				_listener.repository(
					_currentReposId, _currentReposTitle,
					_currentReposReadable, _currentReposWritable);

				_currentReposId = null;
				_currentReposReadable = false;
				_currentReposWritable = false;
				_currentReposTitle = null;
			}
		}
	}
}
