/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * A configuration object containing the configuration parameters for a
 * repository. Configuration options for repositories include read-write
 * permissions and the configuration of the Sail stack. Read- and write
 * permissions can be configured separately as publicly accessible, or to be
 * restricted to specific accounts. A repository that is both publicly readable
 * and publicly writeable is called a 'public repository'. A repository that is
 * neither publicly readable nor publicly writeable is called a 'private
 * repository'.
 *
 * @author Jeen Broekstra, Arjohn Kampman
 */
public class RepositoryConfig implements RepositoryInfo, Cloneable {

/*----------+
| Constants |
+----------*/

	private static final Class RDF_SOURCE_CLASS;
	private static final Class RDF_REPOSITORY_CLASS;
	private static final Class RDF_SCHEMA_SOURCE_CLASS;
	private static final Class RDF_SCHEMA_REPOSITORY_CLASS;

	static {
		try {
			RDF_SOURCE_CLASS = Class.forName("org.openrdf.sesame.sail.RdfSource");
			RDF_REPOSITORY_CLASS = Class.forName("org.openrdf.sesame.sail.RdfRepository");
			RDF_SCHEMA_SOURCE_CLASS = Class.forName("org.openrdf.sesame.sail.RdfSchemaSource");
			RDF_SCHEMA_REPOSITORY_CLASS = Class.forName("org.openrdf.sesame.sail.RdfSchemaRepository");
		}
		catch (ClassNotFoundException e) {
			throw new RuntimeException(e);
		}
	}

/*----------+
| Variables |
+----------*/

	/** The repository's ID. */
	private String _id;

	/** The repository's title. */
	private String _title;

	/** Flag indicating whether the repository should be publicly readable. */
	private boolean _worldReadable;

	/** Flag indicating whether the repository should be publicly writeable. */
	private boolean _worldWriteable;

	/**
	 * List of SailConfig objects, representing the Sail stack from top to
	 * bottom.
	 */
	private List _sailList;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new repository configuration object for a private repository
	 * with the specified ID.
	 *
	 * @param id The repository id.
	 */
	public RepositoryConfig(String id) {
		this(id, null);
	}

	/**
	 * Creates a new repository configuration object for a private repository
	 * with the specified ID and title.
	 *
	 * @param id The repository id.
	 * @param title The repository title.
	 */
	public RepositoryConfig(String id, String title) {
		this(id, title, false, false);
	}

	/**
	 * Creates a new repository configuration object for a repository with the
	 * specified ID and title. Whether or not the repository is publicly
	 * readable and/or writeable is controlled by the <tt>worldReadable</tt> and
	 * <tt>worldWriteable</tt> parameters.
	 *
	 * @param id The repository id.
	 * @param worldReadable Flag indicating wether the repository should be
	 * publicly readable.
	 * @param worldWriteable Flag indicating wether the repository should be
	 * publicly writeable.
	 */
	public RepositoryConfig(String id, boolean worldReadable, boolean worldWriteable) {
		this(id, null, worldReadable, worldWriteable);
	}

	/**
	 * Creates a new repository configuration object for a repository with the
	 * specified ID and title. Whether or not the repository is publicly
	 * readable and/or writeable is controlled by the <tt>worldReadable</tt> and
	 * <tt>worldWriteable</tt> parameters.
	 *
	 * @param id The repository id.
	 * @param title The repository title.
	 * @param worldReadable Flag indicating wether the repository should be
	 * publicly readable.
	 * @param worldWriteable Flag indicating wether the repository should be
	 * publicly writeable.
	 */
	public RepositoryConfig(String id, String title, boolean worldReadable, boolean worldWriteable) {
		_id = id;
		_title = title;
		_worldReadable = worldReadable;
		_worldWriteable = worldWriteable;

		_sailList = new LinkedList();
	}

/*--------+
| Methods |
+--------*/

	/**
	 * Gets the ID of this repository.
	 *
	 * @return The repository id.
	 */
	public String getRepositoryId() {
		return _id;
	}

	/**
	 * Sets the ID of this repository.
	 *
	 * @param id The new repository id.
	 */
	public void setRepositoryId(String id) {
		_id = id;
	}

	/**
	 * Gets the title of the repository.
	 *
	 * @return the repository title
	 */
	public String getTitle() {
		return _title;
	}

	/**
	 * Sets the title of the repository.
	 *
	 * @param title the repository title
	 */
	public void setTitle(String title) {
		_title = title;
	}

	/**
	 * Checks whether this repository is publicly readable.
	 **/
	public boolean isWorldReadable() {
		return _worldReadable;
	}

	/**
	 * Sets the world-readable flag of this repository.
	 **/
	public void setWorldReadable(boolean worldReadable) {
		_worldReadable = worldReadable;
	}

	/**
	 * Checks whether this repository is publicly writeable.
	 **/
	public boolean isWorldWriteable() {
		return _worldWriteable;
	}

	/**
	 * Sets the world-writeable flag of this repository.
	 **/
	public void setWorldWriteable(boolean worldWriteable) {
		_worldWriteable = worldWriteable;
	}

	/**
	 * Makes this repository private, meaning that it will not be publicly
	 * readable or writeable.
	 **/
	public void makePrivate() {
		setWorldReadable(false);
		setWorldWriteable(false);
	}

	/**
	 * Makes this repository public, meaning that it will be publicly readable
	 * and writeable.
	 **/
	public void makePublic() {
		setWorldReadable(true);
		setWorldWriteable(true);
	}

	/**
	 * Checks if this repository is readable.
	 *
	 * @return <tt>true</tt> if the Sail on top of the Sail stack
	 * implements <tt>org.openrdf.sesame.sail.RdfSource</tt>,
	 * <tt>false</tt> otherwise.
	 **/
	public boolean isReadable() {
		if (_sailList.size() > 0) {
			SailConfig topSail = (SailConfig)_sailList.get(0);

			try {
				Class sailClass = Class.forName(topSail.getSailClass());
				return RDF_SOURCE_CLASS.isAssignableFrom(sailClass);
			}
			catch (ClassNotFoundException ignore) {}
		}

		return false;
	}

	/**
	 * Checks if this repository is writeable.
	 *
	 * @return <tt>true</tt> if the Sail on top of the Sail stack
	 * implements <tt>org.openrdf.sesame.sail.RdfRepository</tt>,
	 * <tt>false</tt> otherwise.
	 **/
	public boolean isWriteable() {
		if (_sailList.size() > 0) {
			SailConfig topSail = (SailConfig)_sailList.get(0);

			try {
				Class sailClass = Class.forName(topSail.getSailClass());
				return RDF_REPOSITORY_CLASS.isAssignableFrom(sailClass);
			}
			catch (ClassNotFoundException ignore) {}
		}

		return false;
	}

	/**
	 * Stacks the supplied Sail configuration on top of the current stack of
	 * Sail configurations.
	 *
	 * @param sailConfig A Sail configuration.
	 */
	public void stackSail(SailConfig sailConfig) {
		_sailList.add(0, sailConfig);
	}

	/**
	 * Adds the supplied Sail configuration to the bottom of the current stack
	 * of Sail configurations.
	 *
	 * @param sailConfig A Sail configuration.
	 */
	public void addSail(SailConfig sailConfig) {
		_sailList.add(sailConfig);
	}

	/**
	 * Inserts the supplied Sail configuration into the current stack of Sail
	 * configuration. The sail is inserted in the stack at the position
	 * indicated by the specified index.
	 *
	 * @param sailConfig the SailConfig class
	 * @param index the position in the sail stack for this Sail. The
	 * top-most sail is at index 0.
	 */
	public void addSail(SailConfig sailConfig, int index) {
		_sailList.add(index, sailConfig);
	}

	/**
	 * Removes the supplied Sail configuration from the current stack of Sail
	 * configurations.
	 *
	 * @param sailConfig The Sail configuration to remove.
	 */
	public boolean removeSail(SailConfig sailConfig) {
	 	return _sailList.remove(sailConfig);
	}

	/**
	 * Removes the Sail configuration that is at the specified index from the
	 * current stack of Sail configurations.
	 *
	 * @param index The index of the Sail configuration to remove, 0 for the
	 * top-most Sail configuration.
	 */
	public SailConfig removeSail(int index) {
		return (SailConfig)_sailList.remove(index);
	}

	/**
	 * Retrieves a list containing the current stack of Sail configurations from
	 * top to bottom.
	 *
	 * @return An unmodifiable List of SailConfig objects.
	 * @see SailConfig
	 */
	public List getSailList() {
		return Collections.unmodifiableList(_sailList);
	}

	/**
	 * Sets the Sail stack.
	 *
	 * @param sailList a list of SailConfig objects.
	 */
	public void setSailList(List sailList) {
		_sailList = sailList;
	}

	/**
	 * Checks if any Sails are specified in this RepositoryConfig.
	 */
	public boolean hasASail() {
		return !_sailList.isEmpty();
	}

	/**
	 * Checks if the supplied sail is present in the sail stack.
	 *
	 * @param sailClass the class name of the sail to be searched for.
	 */
	public boolean hasSail(String sailClass) {
		return getSail(sailClass) != null;
	}

	/**
	 * Retrieves the SailConfig object for the supplied sail class.
	 *
	 * @param sailClass the class name of the sail to be retrieved.
	 * @return a SailConfig object.
	 */
	public SailConfig getSail(String sailClass) {
		Iterator iter = _sailList.iterator();

		while (iter.hasNext()) {
			SailConfig sailConfig = (SailConfig)iter.next();

			if (sailConfig.getSailClass().equals(sailClass)) {
				return sailConfig;
			}
		}

		return null;
	}

	/**
	 * Retrieves the interface implemented by the top-most Sail in the
	 * stack. The returned interface is one of the four basic Sail
	 * interfaces: RdfSource, RdfRepository, RdfSchemaSource, and
	 * RdfSchemaRepository.
	 *
	 * @return the interface name
	 */
	public String getSailType() {
		String sailType = null;
		SailConfig topSailConfig = (SailConfig)_sailList.get(0);
		String sailClassName = topSailConfig.getSailClass();

		try {
			Class sailClass = Class.forName(sailClassName);

			if (RDF_SCHEMA_REPOSITORY_CLASS.isAssignableFrom(sailClass)) {
				sailType = "org.openrdf.sesame.sail.RdfSchemaRepository";
			}
			else if (RDF_REPOSITORY_CLASS.isAssignableFrom(sailClass)) {
				sailType = "org.openrdf.sesame.sail.RdfRepository";
			}
			else if (RDF_SCHEMA_SOURCE_CLASS.isAssignableFrom(sailClass)) {
				sailType = "org.openrdf.sesame.sail.RdfSchemaSource";
			}
			else if (RDF_SOURCE_CLASS.isAssignableFrom(sailClass)) {
				sailType = "org.openrdf.sesame.sail.RdfSource";
			}
		}
		catch (ClassNotFoundException ignore) {
		}

		return sailType;
	}

	/**
	 * Moves the sail with the supplied class one level up in the sail stack.
	 *
	 * @param sailClass Sail class
	 */
	public void sailUp(String sailClass) {
		int idx = 0;
		SailConfig sailConfig = null;

		while (idx < _sailList.size()) {
			sailConfig = (SailConfig)_sailList.get(idx);

			if (sailConfig.getSailClass().equals(sailClass)) {
				break;
			}

			idx++;
		}

		if (idx > 0 && idx < _sailList.size()) {
			_sailList.remove(idx);
			_sailList.add(idx - 1, sailConfig);
		}
	}

	/**
	 * Moves the sail with the supplied class one level down in the sail stack.
	 *
	 * @param sailClass Sail class
	 */
	public void sailDown(String sailClass) {
		int idx = 0;
		SailConfig sailConfig = null;

		while (idx < _sailList.size()) {
			sailConfig = (SailConfig)_sailList.get(idx);

			if (sailConfig.getSailClass().equals(sailClass)) {
				break;
			}

			idx++;
		}

		if (idx < _sailList.size() - 1) {
			_sailList.remove(idx);
			_sailList.add(idx + 1, sailConfig);
		}
	}

	// Overrides Object.clone()
	public Object clone() {
		try {
			RepositoryConfig clone = (RepositoryConfig)super.clone();

			// Create a deep-copy of the Sail list
			clone._sailList = new LinkedList();

			Iterator iter = _sailList.iterator();
			while (iter.hasNext()) {
				SailConfig sailConfig = (SailConfig)iter.next();
				clone._sailList.add( (SailConfig)sailConfig.clone() );
			}

			return clone;
		}
		catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
	}
}
