/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.admin;

import java.io.IOException;
import java.io.OutputStream;

import org.openrdf.util.xml.XmlWriter;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

/**
 * Writer for writing administrative messages in XML format. An
 * example administrative message document is:
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;transaction&gt;
 *   &lt;status&gt;
 *     &lt;msg&gt;foobar&lt;/msg&gt;
 *   &lt;/status&gt;
 *   &lt;notification&gt;
 *     &lt;msg&gt;barfoo&lt;/msg&gt;
 *     &lt;line&gt;10&lt;/line&gt;
 *     &lt;column&gt;10&lt;/column&gt;
 *     &lt;statement&gt;
 *       &lt;bNode&gt;a01&lt;/bNode&gt;
 *       &lt;uri&gt;foo:bar&lt;/uri&gt;
 *       &lt;literal xml:lang="en" datatype="bar:foo"&gt;bladibla&lt;/literal&gt;
 *     &lt;/statement&gt;
 *   &lt;/notification&gt;
 * &lt;/transaction&gt;
 * </pre>
 **/
public class XmlAdminMsgWriter implements AdminListener {

/*----------------------------------------------+
| Variables                                     |
+----------------------------------------------*/

	private XmlWriter _xmlWriter;
	protected OutputStream _responseStream;

	private boolean _transactionStarted;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	public XmlAdminMsgWriter(OutputStream responseStream) {
		_responseStream = responseStream;

		_xmlWriter = new XmlWriter(responseStream);
		_xmlWriter.setPrettyPrint(true);

		_transactionStarted = false;
	}

/*----------------------------------------------+
| Methods                                       |
+----------------------------------------------*/

	public void transactionStart() {
		if (_transactionStarted) {
			return;
		}

		try {
			_xmlWriter.startDocument();
			_xmlWriter.startTag("transaction");
		}
		catch (IOException ignore) {
		}
		finally {
			_transactionStarted = true;
		}
	}

	public void transactionEnd() {
		if (!_transactionStarted) {
			return;
		}

		try {
			_xmlWriter.endTag("transaction");
			_xmlWriter.endDocument();

			_responseStream.close();
		}
		catch (IOException ignore) {
		}
		finally {
			_transactionStarted = false;
		}
	}

	public void status(String msg, int lineNo, int colNo) {
		_writeMsgBlock("status", msg, lineNo, colNo, null);
	}

	public void notification(String msg, int lineNo, int colNo, Statement statement) {
		_writeMsgBlock("notification", msg, lineNo, colNo, statement);
	}

	public void warning(String msg, int lineNo, int colNo, Statement statement) {
		_writeMsgBlock("warning", msg, lineNo, colNo, statement);
	}

	public void error(String msg, int lineNo, int colNo, Statement statement) {
		_writeMsgBlock("error", msg, lineNo, colNo, statement);
	}

	private void _writeMsgBlock(String blockType, String msg, int lineNo, int colNo, Statement statement) {
		try {
			_xmlWriter.startTag(blockType);

			_xmlWriter.textElement("msg", msg == null ? "" : msg);

			if (lineNo != -1) {
				_xmlWriter.textElement("line", lineNo);
				if (colNo != -1) {
					_xmlWriter.textElement("column", colNo);
				}
			}

			if (statement != null) {
				_xmlWriter.startTag("statement");

				_writeValue(statement.getSubject());
				_writeValue(statement.getPredicate());
				_writeValue(statement.getObject());

				_xmlWriter.endTag("statement");
			}

			_xmlWriter.endTag(blockType);
		}
		catch (IOException ignore) {
		}
	}

	private void _writeValue(Value value)
		throws IOException
	{
		if (value instanceof Literal) {
			Literal literal = (Literal)value;

			if (literal.getLanguage() != null) {
				_xmlWriter.setAttribute("xml:lang", literal.getLanguage());
			}

			if (literal.getDatatype() != null) {
				_xmlWriter.setAttribute("datatype", literal.getDatatype().getURI());
			}
		}

		if (value instanceof URI) {
			_xmlWriter.textElement("uri", ((URI)value).getURI());
		}
		else if (value instanceof BNode) {
			_xmlWriter.textElement("bNode", ((BNode)value).getID());
		}
		else if (value instanceof Literal) {
			_xmlWriter.textElement("literal", ((Literal)value).getLabel());
		}
	}
}
