/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.admin;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Map;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.xml.SimpleSAXAdapter;
import org.openrdf.util.xml.SimpleSAXParser;

import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

/**
 * Parser for parsing responses from administrative Sesame services.
 * The parser assumes that the format of the acquired data is correct,
 * and doesn't do any extra verification. See the documentation for
 * <tt>XmlAdminMsgWriter</tt> for a description of the XML format.
 *
 * @see XmlAdminMsgWriter
 **/
public class XmlAdminMsgReader {

/*------------------------------+
| Variables                     |
+------------------------------*/

	/**
	 * The SimpleSAXParser that is used for parsing the XML.
	 **/
	private SimpleSAXParser _simpleSAXParser;

	/**
	 * The ValueFactory that is used to create URIs, BNodes and Literals.
	 **/
	private ValueFactory _valueFactory;

/*------------------------------+
| Constructors                  |
+------------------------------*/

	/**
	 * Creates a new XmlAdminMsgReader that will use the supplied
	 * <tt>XMLReader</tt> for parsing the XML admin message documents and an
	 * instance of <tt>ValueFactoryImpl</tt> to create any model objects.
	 **/
	public XmlAdminMsgReader(XMLReader xmlReader) {
		this(xmlReader, new ValueFactoryImpl());
	}

	/**
	 * Creates a new XmlAdminMsgReader that will use the supplied
	 * <tt>XMLReader</tt> for parsing the XML admin message documents and the
	 * supplied <tt>ValueFactory</tt> to create any model objects.
	 **/
	public XmlAdminMsgReader(XMLReader xmlReader, ValueFactory valueFactory) {
		_simpleSAXParser = new SimpleSAXParser(xmlReader);
		_valueFactory = valueFactory;
	}

/*------------------------------+
| Methods                       |
+------------------------------*/

	public synchronized void read(InputStream in, AdminListener listener)
		throws SAXException, IOException
	{
		_simpleSAXParser.setListener(new XmlAdminMsgParser(listener));
		_simpleSAXParser.parse(in);
	}

/*------------------------------+
| Inner class XmlAdminMsgReader |
+------------------------------*/

	class XmlAdminMsgParser extends SimpleSAXAdapter {

		/**
		 * The listener to report the messages to.
		 **/
		private AdminListener _listener;

		private String _currentMsg;

		private int _currentColumnNo = -1;
		private int _currentLineNo = -1;

		private ArrayList _values = new ArrayList(3);

		private Statement _currentStatement;

		public XmlAdminMsgParser(AdminListener listener) {
			_listener = listener;
		}

		public void startTag(String tagName, Map atts, String text) {
			if (tagName.equals("transaction")) {
				_listener.transactionStart();
			}
			/*
			 * messages
			 */
			else if (tagName.equals("msg")) {
				_currentMsg = text;
			}
			/*
			 * line- and column numbers
			 */
			else if (tagName.equals("line")) {
				try {
					_currentLineNo = Integer.parseInt(text);
				}
				catch (NumberFormatException e) {
					_currentLineNo = -1;
				}
			}
			else if (tagName.equals("column")) {
				try {
					_currentColumnNo = Integer.parseInt(text);
				}
				catch (NumberFormatException e) {
					_currentColumnNo = -1;
				}
			}
			/*
			 * uri, bNode, literal
			 */
			else if (tagName.equals("uri")) {
				_values.add(_valueFactory.createURI(text));
			}
			else if (tagName.equals("bNode")) {
				_values.add(_valueFactory.createBNode());
			}
			else if (tagName.equals("literal")) {
				String xmlLang = (String)atts.get("xml:lang");
				String datatype = (String)atts.get("datatype");

				Literal literal = null;
				if (datatype != null) {
					URI dtURI = _valueFactory.createURI(datatype);
					literal = _valueFactory.createLiteral(text, dtURI);
				}
				else if (xmlLang != null) {
					literal = _valueFactory.createLiteral(text, xmlLang);
				}
				else {
					literal = _valueFactory.createLiteral(text);
				}

				_values.add(literal);
			}
		}
	
		public void endTag(String tagName) {
			if (tagName.equals("transaction")) {
				_listener.transactionEnd();
			}
			else if (tagName.equals("statement")) {
				if (_values.size() >= 3) {
					try {
						Resource subject = (Resource)_values.get(0);
						URI predicate = (URI)_values.get(1);
						Value object = (Value)_values.get(2);
						_currentStatement = _valueFactory.createStatement(subject, predicate, object);
					}
					catch (ClassCastException e) {
						// ignore
					}
				}
			}
			/*
			 * status-, notification-, warning- and error messages
			 */
			else if (tagName.equals("status")) {
				_listener.status(_currentMsg, _currentLineNo, _currentColumnNo);
			}
			else if (tagName.equals("notification")) {
				_listener.notification(_currentMsg, _currentLineNo, _currentColumnNo, _currentStatement);
			}
			else if (tagName.equals("warning")) {
				_listener.warning(_currentMsg, _currentLineNo, _currentColumnNo, _currentStatement);
			}
			else if (tagName.equals("error")) {
				_listener.error(_currentMsg, _currentLineNo, _currentColumnNo, _currentStatement);
			}
	
			// Cleaning up
			if (tagName.equals("status") || tagName.equals("notification") ||
			    tagName.equals("warning") || tagName.equals("error"))
			{
				_currentMsg = null;
				_currentLineNo = -1;
				_currentColumnNo = -1;
				_values.clear();
				_currentStatement = null;
			}
		}
	}
}
