/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.admin;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;

import org.openrdf.util.xml.XmlUtil;

import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

public class HtmlAdminMsgWriter implements AdminListener {

/*----------------------------------------------+
| Variables                                     |
+----------------------------------------------*/

	protected OutputStream _responseStream;

	protected PrintWriter _out;

	private boolean _transactionStarted;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	public HtmlAdminMsgWriter(OutputStream responseStream)
		throws IOException
	{
		_responseStream = responseStream;
		_out = new PrintWriter(
				new OutputStreamWriter(responseStream, "UTF-8"));

		_transactionStarted = false;
	}

/*----------------------------------------------+
| Methods                                       |
+----------------------------------------------*/

	public void transactionStart() {
		if (_transactionStarted) {
			return;
		}

		_out.println("<html>");
		_out.println("<head>");
		_out.println("<title>Transaction status</title>");
		_out.println("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">");
		_out.println("</head>");
		_out.println("<body bgcolor=\"white\">");
		_out.println("<pre>");
		_out.println("<font color=\"black\"><b>Transaction started...</b></font>");
		_out.flush();

		_transactionStarted = true;
	}

	public void transactionEnd() {
		if (!_transactionStarted) {
			return;
		}

		_out.println();
		_out.println("<font color=\"black\"><b>Transaction finished</b></font>");
		_out.println("</pre>");
		_out.println("</html>");
		_out.flush();

		_transactionStarted = false;
	}

	public void status(String msg, int lineNo, int colNo) {
		_out.println();
		_out.print("<font size=\"-1\" color=\"black\">");
		_out.print("Status");
		_printLineAndColumn(lineNo, colNo);
		_out.println(":</font>");

		_out.print("<font color=\"black\"><b>");
		if (msg != null) {
			_out.print(XmlUtil.escapeCharacterData(msg));
		}
		else {
			_out.print("(unspecified)");
		}
		_out.println("</b></font>");
		_out.flush();
	}

	public void notification(String msg, int lineNo, int colNo, Statement statement) {
		_out.println();
		_out.print("<font size=\"-1\" color=\"green\">");
		_out.print("Notification");
		_printLineAndColumn(lineNo, colNo);
		_out.println(":</font>");

		_out.print("<font color=\"green\"><b>");
		if (msg != null) {
			_out.print(XmlUtil.escapeCharacterData(msg));
		}
		else {
			_out.print("(unspecified)");
		}
		_out.println("</b></font>");

		_dumpStatement(statement);

		_out.flush();
	}

	public void warning(String msg, int lineNo, int colNo, Statement statement) {
		_out.println();
		_out.print("<font size=\"-1\" color=\"orange\">");
		_out.print("Warning");
		_printLineAndColumn(lineNo, colNo);
		_out.println(":</font>");

		_out.print("<font color=\"orange\"><b>");
		if (msg != null) {
			_out.print(XmlUtil.escapeCharacterData(msg));
		}
		else {
			_out.print("(unspecified)");
		}
		_out.println("</b></font>");

		_dumpStatement(statement);

		_out.flush();
	}

	public void error(String msg, int lineNo, int colNo, Statement statement) {
		_out.println();
		_out.print("<font size=\"-1\" color=\"red\">");
		_out.print("Error");
		_printLineAndColumn(lineNo, colNo);
		_out.println(":</font>");

		_out.print("<font color=\"red\"><b>");
		if (msg != null) {
			_out.print(XmlUtil.escapeCharacterData(msg));
		}
		else {
			_out.print("(unspecified)");
		}
		_out.println("</b></font>");

		_dumpStatement(statement);

		_out.flush();
	}

	private void _printLineAndColumn(int lineNo, int colNo) {
		if (lineNo != -1) {
			_out.print("[line=" + lineNo);
			if (colNo != -1) {
				_out.print("; column=" + colNo);
			}
			_out.print("]");
		}
	}

	private void _dumpStatement(Statement statement) {
		if (statement != null) {
			Resource subject = statement.getSubject();
			_argLine("\tsubject   : " + XmlUtil.escapeCharacterData(subject.toString()));

			URI predicate = statement.getPredicate();
			_argLine("\tpredicate : " + XmlUtil.escapeCharacterData(predicate.toString()));

			Value object = statement.getObject();
			if (object instanceof Literal) {
				_argLine("\tobject    : <i>" + XmlUtil.escapeCharacterData(object.toString()) + "</i>");
			}
			else {
				_argLine("\tobject    : " + XmlUtil.escapeCharacterData(object.toString()));
			}
		}
	}

	private void _argLine(String msg) {
		_out.print("<font size=\"-1\" color=\"black\">");
		_out.print(msg);
		_out.println("</font>");
	}
}
