/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.admin;

import java.util.ArrayList;
import java.util.List;

import org.openrdf.model.Statement;

/**
 * An implementation of the AdminListener interface that collects the
 * reported messages and allows them to be checked after the performed
 * action has (sucessfully or unsuccessfully) been completed.
 **/
public class AdminMsgCollector implements AdminListener {

/*--------------------------------------------+
| Variables                                   |
+--------------------------------------------*/

	/** The list of collected messages. **/
	private List _messages = new ArrayList();

/*--------------------------------------------+
| Constructors                                |
+--------------------------------------------*/

	/**
	 * Creates a new AdminMsgCollector object.
	 **/
	public AdminMsgCollector() {
	}

/*--------------------------------------------+
| Methods                                     |
+--------------------------------------------*/

	/**
	 * Gets all collected messages.
	 * @return a List of AdminMsg objects.
	 * @see AdminMsg
	 **/
	public List getMessages() {
		return new ArrayList(_messages);
	}

	/**
	 * Return <tt>true</tt> if any errors have been collected, false
	 * otherwise.
	 **/
	public boolean hasErrors() {
		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.ERROR) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Gets all collected error messages.
	 * @return a List of AdminMsg objects of type ERROR.
	 * @see AdminMsg
	 **/
	public List getErrors() {
		ArrayList errors = new ArrayList();

		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.ERROR) {
				errors.add(adminMsg);
			}
		}

		return errors;
	}

	/**
	 * Return <tt>true</tt> if any warnings have been collected, false
	 * otherwise.
	 **/
	public boolean hasWarnings() {
		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.WARNING) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Gets all collected warnings.
	 * @return a List of AdminMsg objects of type WARNING.
	 * @see AdminMsg
	 **/
	public List getWarnings() {
		ArrayList warnings = new ArrayList();

		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.WARNING) {
				warnings.add(adminMsg);
			}
		}

		return warnings;
	}

	/**
	 * Return <tt>true</tt> if any notifications have been collected, false
	 * otherwise.
	 **/
	public boolean hasNotifications() {
		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.NOTIFICATION) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Gets all collected notifications.
	 * @return a List of AdminMsg objects of type NOTIFICATION.
	 * @see AdminMsg
	 **/
	public List getNotifications() {
		ArrayList notifications = new ArrayList();

		for (int i = 0; i < _messages.size(); i++) {
			AdminMsg adminMsg = (AdminMsg)_messages.get(i);
			if (adminMsg.getType() == AdminMsg.NOTIFICATION) {
				notifications.add(adminMsg);
			}
		}

		return notifications;
	}

/*--------------------------------------------+
| Methods from interface AdminListener        |
+--------------------------------------------*/

	public void transactionStart() {
		_messages.clear();
	}

	public void transactionEnd() {
	}

	public void status(String msg, int lineNo, int columnNo)
	{
		_addMessage(AdminMsg.STATUS, msg, lineNo, columnNo);
	}

	public void notification(
		String msg, int lineNo, int columnNo, Statement stat)
	{
		_addMessage(AdminMsg.NOTIFICATION, msg, lineNo, columnNo);
	}

	public void warning(
		String msg, int lineNo, int columnNo, Statement stat)
	{
		_addMessage(AdminMsg.WARNING, msg, lineNo, columnNo);
	}

	public void error(
		String msg, int lineNo, int columnNo, Statement stat)
	{
		_addMessage(AdminMsg.ERROR, msg, lineNo, columnNo);
	}

	private void _addMessage(int msgType, String msg, int lineNo, int columnNo) {
		_messages.add( new AdminMsg(msgType, msg, lineNo, columnNo) );
	}
}
