/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.rmi.Naming;

import org.openrdf.util.rmirouting.ChannelIfaceInvocation;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.handlers.SystemConfigFileHandler;
import org.openrdf.sesame.repository.SesameService;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.repository.remote.HTTPService;
import org.openrdf.sesame.server.rmi.FactoryInterface;

/**
 * Class providing a static entry point to the Sesame APIs.
 *
 * @author Herko ter Horst
 * @author Arjohn Kampman
 */
public class Sesame {

	/**
	 * Get an empty local SesameService.
	 *
	 * @return An empty LocalService.
	 */
	public static LocalService getService() {
		return new LocalService();
	}

	/**
	 * Get a local SesameService initialized with the supplied configuration.
	 *
	 * @param config The configuration.
	 * @return A LocalService objet.
	 */
	public static LocalService getService(SystemConfig config) {
		return new LocalService(config);
	}

	/**
	 * Get a local SesameService initialized with the specified system
	 * configuration file. Please make sure that the system property
	 * <tt>org.xml.sax.driver</tt> has been set and is poiting to an
	 * implementation of <tt>org.xml.sax.XMLReader</tt>. If this property is not
	 * set, the specified file cannot be read.
	 *
	 * @param systemConfigFile The system configuration file.
	 * @return A LocalService object.
	 * @exception IOException If the specified system configuration file could
	 * not be read.
	 **/
	public static LocalService getService(File systemConfigFile)
		throws IOException
	{
		InputStream in = new FileInputStream(systemConfigFile);
		try {
			SystemConfig sysConfig = SystemConfigFileHandler.readConfiguration(in);
			return getService(sysConfig);
		}
		finally {
			in.close();
		}
	}

	/**
	 * Get a remote SesameService object for a server running at the specified
	 * URL. This method can be used to retrieve a Sesame connection over
	 * HTTP.
	 *
	 * @param serverURL The URL to connect to, e.g.
	 * <tt>http://localhost:8080/sesame/</tt>
	 * @return An HTTPService object for the specified server.
	 * @exception IllegalArgumentException If the protocol of the specified URL
	 * is not "http".
	 */
	public static HTTPService getService(URL serverURL) {
		return new HTTPService(serverURL);
	}

	/**
	 * Get a remote SesameService object for a server running at the specified
	 * URI. This method can be used to retrieve a Sesame connection
	 * through RMI.
	 *
	 * @param serverURI The URI to connect to, e.g.
	 * <tt>rmi://remote.sesame.svr:1099/</tt>.
	 * @return An SesameService object for the specified server.
	 */
	public static SesameService getService(URI serverURI) {
		if (serverURI.getScheme().equals("rmi")) {
			String s = serverURI.toString();
			if (s.endsWith("/")) {
				s = s + "FactoryInterface";
			}
			else {
				s = s + "/FactoryInterface";
			}
			
			try {
				FactoryInterface fi = (FactoryInterface)ChannelIfaceInvocation.wrapIt(Naming.lookup(s));
				return fi.getService();
			}
			catch (Throwable t) {
				t.printStackTrace();
				throw new RuntimeException("Unable to connect to Sesame service at '" + serverURI, t);
			}
		}
		else {
			throw new IllegalArgumentException("URI scheme not supported: " + serverURI.getScheme());
		}
	}
}
