/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.jvnet.jaxb.reflection.runtime.property;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import org.jvnet.jaxb.reflection.impl.api.AccessorException;
import org.jvnet.jaxb.reflection.model.core.PropertyKind;
import org.jvnet.jaxb.reflection.model.core.TypeRef;
import org.jvnet.jaxb.reflection.model.runtime.RuntimeElementPropertyInfo;
import org.jvnet.jaxb.reflection.model.runtime.RuntimeTypeInfo;
import org.jvnet.jaxb.reflection.model.runtime.RuntimeTypeRef;
import org.jvnet.jaxb.reflection.runtime.JAXBContextImpl;
import org.jvnet.jaxb.reflection.runtime.JaxBeanInfo;
import org.jvnet.jaxb.reflection.runtime.Name;
import org.jvnet.jaxb.reflection.runtime.XMLSerializer;
import org.jvnet.jaxb.reflection.runtime.reflect.Accessor;
import org.jvnet.jaxb.reflection.runtime.unmarshaller.ChildLoader;
import org.jvnet.jaxb.reflection.runtime.unmarshaller.DefaultValueLoaderDecorator;
import org.jvnet.jaxb.reflection.runtime.unmarshaller.Loader;
import org.jvnet.jaxb.reflection.runtime.unmarshaller.XsiNilLoader;
import org.jvnet.jaxb.reflection.util.QNameMap;

import org.xml.sax.SAXException;

/**
 * @author Kohsuke Kawaguchi (kk@kohsuke.org)
 */
final class SingleElementNodeProperty<BeanT,ValueT> extends PropertyImpl<BeanT> {

    private final Accessor<BeanT,ValueT> acc;

    private final boolean nillable;

    private final QName[] acceptedElements;

    private final Map<Class,TagAndType> typeNames = new HashMap<Class,TagAndType>();

    private RuntimeElementPropertyInfo prop;

    /**
     * The tag name used to produce xsi:nil. The first one in the list.
     */
    private final Name nullTagName;

    public SingleElementNodeProperty(JAXBContextImpl context, RuntimeElementPropertyInfo prop) {
        super(context,prop);
        acc = prop.getAccessor().optimize(context);
        this.prop = prop;

        QName nt = null;
        boolean nil = false;

        acceptedElements = new QName[prop.getTypes().size()];
        for( int i=0; i<acceptedElements.length; i++ )
            acceptedElements[i] = prop.getTypes().get(i).getTagName();

        for (RuntimeTypeRef e : prop.getTypes()) {
            JaxBeanInfo beanInfo = context.getOrCreate(e.getTarget());
            if(nt==null)    nt = e.getTagName();
            typeNames.put( beanInfo.jaxbType, new TagAndType(
                context.nameBuilder.createElementName(e.getTagName()),beanInfo) );
            nil |= e.isNillable();
        }

        nullTagName = context.nameBuilder.createElementName(nt);

        nillable = nil;
    }

    public void wrapUp() {
        super.wrapUp();
        prop = null;
    }

    public void reset(BeanT bean) throws AccessorException {
        acc.set(bean,null);
    }

    public String getIdValue(BeanT beanT) {
        return null;
    }

    public void serializeBody(BeanT o, XMLSerializer w, Object outerPeer) throws SAXException, AccessorException, IOException, XMLStreamException {
        ValueT v = acc.get(o);
        if(v!=null) {
            Class vtype = v.getClass();
            TagAndType tt=typeNames.get(vtype); // quick way that usually works

            if(tt==null) {// slow way that always works
                for (Map.Entry<Class,TagAndType> e : typeNames.entrySet()) {
                    if(e.getKey().isAssignableFrom(vtype)) {
                        tt = e.getValue();
                        break;
                    }
                }
            }

            if(tt==null) {
                // actually this is an error, because the actual type was not a sub-type
                // of any of the types specified in the annotations,
                // but for the purpose of experimenting with simple type substitution,
                // it's convenient to marshal this anyway (for example so that classes
                // generated from simple types like String can be marshalled as expected.)
                w.startElement(typeNames.values().iterator().next().tagName,null);
                w.childAsXsiType(v,fieldName,w.grammar.getBeanInfo(Object.class));
            } else {
                w.startElement(tt.tagName,null);
                w.childAsXsiType(v,fieldName,tt.beanInfo);
            }
            w.endElement();
        } else
        if(nillable) {
            w.startElement(nullTagName,null);
            w.writeXsiNilTrue();
            w.endElement();
        }
    }

    public void buildChildElementUnmarshallers(UnmarshallerChain chain, QNameMap<ChildLoader> handlers) {
        JAXBContextImpl context = chain.context;

        for (TypeRef<Type,Class> e : prop.getTypes()) {
            JaxBeanInfo bi = context.getOrCreate((RuntimeTypeInfo) e.getTarget());
            // if the expected Java type is already final, type substitution won't really work anyway.
            // this also traps cases like trying to substitute xsd:long element with xsi:type='xsd:int'
            Loader l = bi.getLoader(context,!Modifier.isFinal(bi.jaxbType.getModifiers()));
            if(e.getDefaultValue()!=null)
                l = new DefaultValueLoaderDecorator(l,e.getDefaultValue());
            if(nillable || chain.context.allNillable)
                l = new XsiNilLoader.Single(l,acc);
            handlers.put( e.getTagName(), new ChildLoader(l,acc));
        }
    }

    public PropertyKind getKind() {
        return PropertyKind.ELEMENT;
    }

    @Override
    public Accessor getElementPropertyAccessor(String nsUri, String localName) {
        for( QName n : acceptedElements) {
            if(n.getNamespaceURI().equals(nsUri) && n.getLocalPart().equals(localName))
                return acc;
        }
        return null;
    }

}
