/*
 * Decompiled with CFR 0.152.
 */
package org.openrdf.sesame.sailimpl.nativerdf.btree;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Random;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTreeIterator;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTreeValueComparator;
import org.openrdf.sesame.sailimpl.nativerdf.btree.DefaultBTreeValueComparator;
import org.openrdf.util.ByteArrayUtil;

public class BTree {
    private static final int FILE_FORMAT_VERSION = 1;
    private static final int HEADER_LENGTH = 16;
    private static final int MRU_CACHE_SIZE = 8;
    private File _file;
    private RandomAccessFile _raf;
    private FileChannel _fileChannel;
    private BTreeValueComparator _comparator;
    private int _blockSize;
    private int _valueSize;
    private int _rootNodeID;
    private int _maxNodeID;
    private int _slotSize;
    private int _branchFactor;
    private int _minValueCount;
    private int _nodeSize;
    private LinkedList _nodesInUse;
    private LinkedList _mruNodes;

    public BTree(File dataFile, int blockSize, int valueSize) throws IOException {
        this(dataFile, blockSize, valueSize, new DefaultBTreeValueComparator());
    }

    public BTree(File dataFile, int blockSize, int valueSize, BTreeValueComparator comparator) throws IOException {
        if (blockSize < 16) {
            throw new IllegalArgumentException("block size must be at least 16 bytes");
        }
        if (valueSize <= 0) {
            throw new IllegalArgumentException("value size must be larger than 0");
        }
        if (blockSize < 3 * valueSize + 20) {
            throw new IllegalArgumentException("block size to small; must at least be able to store three values");
        }
        this._file = dataFile;
        this._comparator = comparator;
        this._file.createNewFile();
        this._raf = new RandomAccessFile(this._file, "rw");
        this._fileChannel = this._raf.getChannel();
        if (this._fileChannel.size() == 0L) {
            this._blockSize = blockSize;
            this._valueSize = valueSize;
            this._rootNodeID = 0;
            this._maxNodeID = 0;
            this._writeFileHeader();
        } else {
            this._readFileHeader();
            this._maxNodeID = this._offset2nodeID(this._fileChannel.size() - (long)this._nodeSize);
        }
        this._slotSize = 4 + this._valueSize;
        this._branchFactor = 1 + (this._blockSize - 8) / this._slotSize;
        this._minValueCount = (this._branchFactor - 1) / 2;
        this._nodeSize = 8 + (this._branchFactor - 1) * this._slotSize;
        this._nodesInUse = new LinkedList();
        this._mruNodes = new LinkedList();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws IOException {
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            this._nodesInUse.clear();
        }
        linkedList = this._mruNodes;
        synchronized (linkedList) {
            this._mruNodes.clear();
        }
        this._raf.close();
        this._fileChannel = null;
        this._raf = null;
        this._file = null;
    }

    public void startTransaction() throws IOException {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void commitTransaction() throws IOException {
        Node node;
        Iterator iter2;
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            iter2 = this._nodesInUse.iterator();
            while (iter2.hasNext()) {
                node = (Node)iter2.next();
                if (!node.dataChanged()) continue;
                node.write();
            }
        }
        linkedList = this._mruNodes;
        synchronized (linkedList) {
            iter2 = this._mruNodes.iterator();
            while (iter2.hasNext()) {
                node = (Node)iter2.next();
                if (!node.dataChanged()) continue;
                node.write();
            }
        }
        this._fileChannel.force(false);
    }

    public byte[] get(byte[] key) throws IOException {
        if (this._rootNodeID == 0) {
            return null;
        }
        Node node = this._readNode(this._rootNodeID);
        while (true) {
            int valueIdx;
            if ((valueIdx = node.search(key)) >= 0) {
                byte[] result2 = node.getValue(valueIdx);
                node.release();
                return result2;
            }
            if (node.isLeaf()) break;
            Node childNode = node.getChildNode(-valueIdx - 1);
            node.release();
            node = childNode;
        }
        node.release();
        return null;
    }

    public BTreeIterator iterateAll() {
        return new SeqScanIterator(null, null);
    }

    public BTreeIterator iterateRange(byte[] minValue, byte[] maxValue) {
        return new RangeIterator(null, null, minValue, maxValue);
    }

    public BTreeIterator iterateValues(byte[] searchKey, byte[] searchMask) {
        return new SeqScanIterator(searchKey, searchMask);
    }

    public BTreeIterator iterateValues(byte[] searchKey, byte[] searchMask, byte[] minValue, byte[] maxValue) {
        return new RangeIterator(searchKey, searchMask, minValue, maxValue);
    }

    public byte[] insert(byte[] value2) throws IOException {
        Node rootNode = null;
        if (this._rootNodeID == 0) {
            rootNode = this._createNewNode();
            this._rootNodeID = rootNode.getID();
            this._writeFileHeader();
        } else {
            rootNode = this._readNode(this._rootNodeID);
        }
        InsertResult insertResult = this._insertInTree(value2, 0, rootNode);
        if (insertResult.overflowValue != null) {
            Node newRootNode = this._createNewNode();
            newRootNode.setChildNodeID(0, rootNode.getID());
            newRootNode.insertValueNodeIDPair(0, insertResult.overflowValue, insertResult.overflowNodeID);
            this._rootNodeID = newRootNode.getID();
            this._writeFileHeader();
            newRootNode.release();
        }
        rootNode.release();
        return insertResult.oldValue;
    }

    private InsertResult _insertInTree(byte[] value2, int nodeID, Node node) throws IOException {
        InsertResult insertResult = null;
        int valueIdx = node.search(value2);
        if (valueIdx >= 0) {
            insertResult = new InsertResult();
            insertResult.oldValue = node.getValue(valueIdx);
            if (!Arrays.equals(value2, insertResult.oldValue)) {
                node.setValue(valueIdx, value2);
            }
        } else {
            valueIdx = -valueIdx - 1;
            if (node.isLeaf()) {
                insertResult = this._insertInNode(value2, nodeID, valueIdx, node);
            } else {
                Node childNode = node.getChildNode(valueIdx);
                insertResult = this._insertInTree(value2, nodeID, childNode);
                childNode.release();
                if (insertResult.overflowValue != null) {
                    byte[] oldValue = insertResult.oldValue;
                    insertResult = this._insertInNode(insertResult.overflowValue, insertResult.overflowNodeID, valueIdx, node);
                    insertResult.oldValue = oldValue;
                }
            }
        }
        return insertResult;
    }

    private InsertResult _insertInNode(byte[] value2, int nodeID, int valueIdx, Node node) throws IOException {
        InsertResult insertResult = new InsertResult();
        if (node.isFull()) {
            Node newNode = this._createNewNode();
            insertResult.overflowValue = node.splitAndInsert(value2, nodeID, valueIdx, newNode);
            insertResult.overflowNodeID = newNode.getID();
            newNode.release();
        } else {
            node.insertValueNodeIDPair(valueIdx, value2, nodeID);
        }
        return insertResult;
    }

    public byte[] remove(byte[] key) throws IOException {
        byte[] result2 = null;
        if (this._rootNodeID != 0) {
            Node rootNode = this._readNode(this._rootNodeID);
            result2 = this._removeFromTree(key, rootNode);
            if (result2 != null && rootNode.isEmpty()) {
                Node newRootNode = rootNode.getChildNode(0);
                newRootNode.clearParentInfo();
                this._rootNodeID = newRootNode.getID();
                this._writeFileHeader();
                newRootNode.release();
                rootNode.setChildNodeID(0, 0);
            }
            rootNode.release();
        }
        return result2;
    }

    private byte[] _removeFromTree(byte[] key, Node node) throws IOException {
        byte[] value2 = null;
        int valueIdx = node.search(key);
        if (valueIdx >= 0) {
            if (node.isLeaf()) {
                value2 = node.removeValueRight(valueIdx);
            } else {
                value2 = node.getValue(valueIdx);
                Node rightChildNode = node.getChildNode(valueIdx + 1);
                byte[] smallestValue = this._removeSmallestValueFromTree(rightChildNode);
                node.setValue(valueIdx, smallestValue);
                this._balanceChildNode(node, rightChildNode, valueIdx + 1);
                rightChildNode.release();
            }
        } else if (!node.isLeaf()) {
            valueIdx = -valueIdx - 1;
            Node childNode = node.getChildNode(valueIdx);
            value2 = this._removeFromTree(key, childNode);
            this._balanceChildNode(node, childNode, valueIdx);
            childNode.release();
        }
        return value2;
    }

    private byte[] _removeSmallestValueFromTree(Node node) throws IOException {
        byte[] value2 = null;
        if (node.isLeaf() && !node.isEmpty()) {
            value2 = node.removeValueLeft(0);
        } else {
            Node childNode = node.getChildNode(0);
            value2 = this._removeSmallestValueFromTree(childNode);
            this._balanceChildNode(node, childNode, 0);
            childNode.release();
        }
        return value2;
    }

    private void _balanceChildNode(Node parentNode, Node childNode, int childIdx) throws IOException {
        if (childNode.getValueCount() < this._minValueCount) {
            Node rightSibling;
            Node node = rightSibling = childIdx < parentNode.getValueCount() ? parentNode.getChildNode(childIdx + 1) : null;
            if (rightSibling != null && rightSibling.getValueCount() > this._minValueCount) {
                childNode.insertValueNodeIDPair(childNode.getValueCount(), parentNode.getValue(childIdx), rightSibling.getChildNodeID(0));
                parentNode.setValue(childIdx, rightSibling.removeValueLeft(0));
            } else {
                Node leftSibling;
                Node node2 = leftSibling = childIdx > 0 ? parentNode.getChildNode(childIdx - 1) : null;
                if (leftSibling != null && leftSibling.getValueCount() > this._minValueCount) {
                    childNode.insertNodeIDValuePair(0, leftSibling.getChildNodeID(leftSibling.getValueCount()), parentNode.getValue(childIdx - 1));
                    parentNode.setValue(childIdx - 1, leftSibling.removeValueRight(leftSibling.getValueCount() - 1));
                } else if (leftSibling != null) {
                    leftSibling.mergeWithRightSibling(parentNode.removeValueRight(childIdx - 1), childNode);
                } else {
                    childNode.mergeWithRightSibling(parentNode.removeValueRight(childIdx), rightSibling);
                }
                if (leftSibling != null) {
                    leftSibling.release();
                }
            }
            if (rightSibling != null) {
                rightSibling.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() throws IOException {
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            this._nodesInUse.clear();
        }
        linkedList = this._mruNodes;
        synchronized (linkedList) {
            this._mruNodes.clear();
        }
        this._fileChannel.truncate(16L);
        this._rootNodeID = 0;
        this._maxNodeID = 0;
        this._writeFileHeader();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Node _createNewNode() throws IOException {
        Node node = new Node(++this._maxNodeID);
        node.use();
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            this._nodesInUse.add(node);
        }
        return node;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Node _readNode(int id) throws IOException {
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            Object node;
            Iterator iter2 = this._nodesInUse.iterator();
            while (iter2.hasNext()) {
                node = (Node)iter2.next();
                if (((Node)node).getID() != id) continue;
                ((Node)node).use();
                return node;
            }
            node = this._mruNodes;
            synchronized (node) {
                iter2 = this._mruNodes.iterator();
                while (iter2.hasNext()) {
                    Node node2 = (Node)iter2.next();
                    if (node2.getID() != id) continue;
                    iter2.remove();
                    this._nodesInUse.add(node2);
                    node2.use();
                    return node2;
                }
            }
            node = new Node(id);
            ((Node)node).read();
            this._nodesInUse.add(node);
            ((Node)node).use();
            return node;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void _releaseNode(Node node) throws IOException {
        LinkedList linkedList = this._nodesInUse;
        synchronized (linkedList) {
            this._nodesInUse.remove(node);
            LinkedList linkedList2 = this._mruNodes;
            synchronized (linkedList2) {
                Node lruNode;
                if (this._mruNodes.size() >= 8 && (lruNode = (Node)this._mruNodes.removeLast()).dataChanged()) {
                    lruNode.write();
                }
                this._mruNodes.addFirst(node);
            }
        }
    }

    private long _nodeID2offset(int id) {
        return (long)this._blockSize * (long)id;
    }

    private int _offset2nodeID(long offset) {
        return (int)(offset / (long)this._blockSize);
    }

    private void _writeFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.putInt(1);
        buf.putInt(this._blockSize);
        buf.putInt(this._valueSize);
        buf.putInt(this._rootNodeID);
        buf.rewind();
        this._fileChannel.write(buf, 0L);
    }

    private void _readFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        this._fileChannel.read(buf, 0L);
        buf.rewind();
        int fileFormatVersion = buf.getInt();
        this._blockSize = buf.getInt();
        this._valueSize = buf.getInt();
        this._rootNodeID = buf.getInt();
    }

    public static void main(String[] args) throws Exception {
        System.out.println("Running BTree test...");
        if (args.length > 1) {
            BTree.runPerformanceTest(args);
        } else {
            BTree.runDebugTest(args);
        }
        System.out.println("Done.");
    }

    public static void runPerformanceTest(String[] args) throws Exception {
        File dataFile = new File(args[0]);
        int valueCount = Integer.parseInt(args[1]);
        DefaultBTreeValueComparator comparator = new DefaultBTreeValueComparator();
        BTree btree = new BTree(dataFile, 501, 13, comparator);
        Random random = new Random(0L);
        byte[] value2 = new byte[13];
        long startTime = System.currentTimeMillis();
        for (int i = 1; i <= valueCount; ++i) {
            random.nextBytes(value2);
            btree.insert(value2);
            if (i % 50000 != 0) continue;
            System.out.println("Inserted " + i + " values in " + (System.currentTimeMillis() - startTime) + " ms");
        }
        System.out.println("Iterating over all values in sequential order...");
        startTime = System.currentTimeMillis();
        BTreeIterator iter2 = btree.iterateAll();
        value2 = iter2.next();
        int count2 = 0;
        while (value2 != null) {
            ++count2;
            value2 = iter2.next();
        }
        System.out.println("Iteration over " + count2 + " items finished in " + (System.currentTimeMillis() - startTime) + " ms");
    }

    public static void runDebugTest(String[] args) throws Exception {
        File dataFile = new File(args[0]);
        BTree btree = new BTree(dataFile, 28, 1);
        btree.print(System.out);
    }

    public void print(PrintStream out) throws IOException {
        out.println("---contents of BTree file---");
        out.println("Stored parameters:");
        out.println("block size   = " + this._blockSize);
        out.println("value size   = " + this._valueSize);
        out.println("root node ID = " + this._rootNodeID);
        out.println();
        out.println("Derived parameters:");
        out.println("slot size       = " + this._slotSize);
        out.println("branch factor   = " + this._branchFactor);
        out.println("min value count = " + this._minValueCount);
        out.println("node size       = " + this._nodeSize);
        out.println("max node ID     = " + this._maxNodeID);
        out.println();
        ByteBuffer buf = ByteBuffer.allocate(this._nodeSize);
        for (long offset = (long)this._blockSize; offset < this._fileChannel.size(); offset += (long)this._blockSize) {
            this._fileChannel.read(buf, offset);
            buf.rewind();
            int nodeID = this._offset2nodeID(offset);
            int count2 = buf.getInt();
            out.print("node " + nodeID + ": ");
            out.print("count=" + count2 + " ");
            byte[] value2 = new byte[this._valueSize];
            for (int i = 0; i < count2; ++i) {
                out.print(buf.getInt());
                buf.get(value2);
                out.print("[" + ByteArrayUtil.toHexString((byte[])value2) + "]");
            }
            out.println(buf.getInt());
            buf.clear();
        }
        out.println("---end of BTree file---");
    }

    private class RangeIterator
    implements BTreeIterator {
        private byte[] _searchKey;
        private byte[] _searchMask;
        private byte[] _minValue;
        private byte[] _maxValue;
        private boolean _started;
        private Node _currentNode;
        private int _currentIdx;

        public RangeIterator(byte[] searchKey, byte[] searchMask, byte[] minValue, byte[] maxValue) {
            this._searchKey = searchKey;
            this._searchMask = searchMask;
            this._minValue = minValue;
            this._maxValue = maxValue;
            this._started = false;
        }

        public byte[] next() throws IOException {
            if (!this._started) {
                this._started = true;
                this._findMinimum();
            }
            byte[] value2 = this._findNext(false);
            while (value2 != null) {
                if (this._maxValue != null && BTree.this._comparator.compareBTreeValues(this._maxValue, value2, 0, value2.length) < 0) {
                    while (this._currentNode != null) {
                        Node parentNode = this._currentNode.getParentNode();
                        this._currentNode.release();
                        this._currentNode = parentNode;
                    }
                    value2 = null;
                    break;
                }
                if (this._searchKey == null || ByteArrayUtil.matchesPattern((byte[])value2, (byte[])this._searchMask, (byte[])this._searchKey)) break;
                value2 = this._findNext(false);
            }
            return value2;
        }

        private void _findMinimum() throws IOException {
            if (BTree.this._rootNodeID == 0) {
                return;
            }
            this._currentNode = BTree.this._readNode(BTree.this._rootNodeID);
            while (true) {
                if (this._minValue != null) {
                    this._currentIdx = this._currentNode.search(this._minValue);
                    if (this._currentIdx >= 0) break;
                    this._currentIdx = -this._currentIdx - 1;
                }
                if (this._currentNode.isLeaf()) break;
                this._currentNode = this._currentNode.getChildNode(this._currentIdx);
                this._currentIdx = 0;
            }
        }

        private byte[] _findNext(boolean returnedFromRecursion) throws IOException {
            if (this._currentNode == null) {
                return null;
            }
            if (returnedFromRecursion || this._currentNode.isLeaf()) {
                if (this._currentIdx >= this._currentNode.getValueCount()) {
                    Node parentNode = this._currentNode.getParentNode();
                    this._currentIdx = this._currentNode.getIndexInParent();
                    this._currentNode.release();
                    this._currentNode = parentNode;
                    return this._findNext(true);
                }
                return this._currentNode.getValue(this._currentIdx++);
            }
            this._currentNode = this._currentNode.getChildNode(this._currentIdx);
            this._currentIdx = 0;
            return this._findNext(false);
        }

        public void close() throws IOException {
            if (this._currentNode != null) {
                this._currentNode.release();
                this._currentNode = null;
            }
        }
    }

    private class SeqScanIterator
    implements BTreeIterator {
        private byte[] _searchKey;
        private byte[] _searchMask;
        private int _currentNodeID;
        private Node _currentNode;
        private int _currentIdx;

        public SeqScanIterator(byte[] searchKey, byte[] searchMask) {
            this._searchKey = searchKey;
            this._searchMask = searchMask;
        }

        public byte[] next() throws IOException {
            while (this._currentNodeID <= BTree.this._maxNodeID) {
                if (this._currentNode == null) {
                    this._currentNodeID = 1;
                    this._currentNode = BTree.this._readNode(this._currentNodeID);
                    this._currentIdx = 0;
                }
                while (this._currentIdx < this._currentNode.getValueCount()) {
                    byte[] value2 = this._currentNode.getValue(this._currentIdx++);
                    if (this._searchKey != null && !ByteArrayUtil.matchesPattern((byte[])value2, (byte[])this._searchMask, (byte[])this._searchKey)) continue;
                    return value2;
                }
                this._currentNode.release();
                ++this._currentNodeID;
                this._currentNode = this._currentNodeID <= BTree.this._maxNodeID ? BTree.this._readNode(this._currentNodeID) : null;
                this._currentIdx = 0;
            }
            return null;
        }

        public void close() throws IOException {
            if (this._currentNode != null) {
                this._currentNodeID = BTree.this._maxNodeID + 1;
                this._currentNode.release();
                this._currentNode = null;
            }
        }
    }

    private class Node {
        private int _id;
        private long _offset;
        private byte[] _data;
        private int _valueCount;
        private int _usageCount;
        private boolean _dataChanged;
        private Node _parent;
        private int _indexInParent;

        public Node(int id) {
            this._id = id;
            this._offset = BTree.this._nodeID2offset(this._id);
            this._valueCount = 0;
            this._usageCount = 0;
            this._data = new byte[BTree.this._nodeSize + BTree.this._slotSize];
        }

        public int getID() {
            return this._id;
        }

        public boolean isLeaf() {
            return ByteArrayUtil.getInt((byte[])this._data, (int)4) == 0;
        }

        public void use() {
            ++this._usageCount;
        }

        public void release() throws IOException {
            --this._usageCount;
            if (this._usageCount == 0) {
                BTree.this._releaseNode(this);
            }
        }

        public int getUsageCount() {
            return this._usageCount;
        }

        public boolean dataChanged() {
            return this._dataChanged;
        }

        public int getValueCount() {
            return this._valueCount;
        }

        public boolean isEmpty() {
            return this._valueCount == 0;
        }

        public boolean isFull() {
            return this._valueCount == BTree.this._branchFactor - 1;
        }

        public byte[] getValue(int valueIdx) {
            return ByteArrayUtil.get((byte[])this._data, (int)this._valueIdx2offset(valueIdx), (int)BTree.this._valueSize);
        }

        public void setValue(int valueIdx, byte[] value2) {
            ByteArrayUtil.put((byte[])value2, (byte[])this._data, (int)this._valueIdx2offset(valueIdx));
            this._dataChanged = true;
        }

        public byte[] removeValueRight(int valueIdx) {
            byte[] value2 = this.getValue(valueIdx);
            int endOffset = this._valueIdx2offset(this._valueCount);
            if (valueIdx < this._valueCount - 1) {
                this._shiftData(this._valueIdx2offset(valueIdx + 1), endOffset, -BTree.this._slotSize);
            }
            this._clearData(endOffset - BTree.this._slotSize, endOffset);
            this._setValueCount(--this._valueCount);
            this._dataChanged = true;
            return value2;
        }

        public byte[] removeValueLeft(int valueIdx) {
            byte[] value2 = this.getValue(valueIdx);
            int endOffset = this._valueIdx2offset(this._valueCount);
            this._shiftData(this._nodeIdx2offset(valueIdx + 1), endOffset, -BTree.this._slotSize);
            this._clearData(endOffset - BTree.this._slotSize, endOffset);
            this._setValueCount(--this._valueCount);
            this._dataChanged = true;
            return value2;
        }

        public int getChildNodeID(int nodeIdx) {
            return ByteArrayUtil.getInt((byte[])this._data, (int)this._nodeIdx2offset(nodeIdx));
        }

        public void setChildNodeID(int nodeIdx, int nodeID) {
            ByteArrayUtil.putInt((int)nodeID, (byte[])this._data, (int)this._nodeIdx2offset(nodeIdx));
            this._dataChanged = true;
        }

        public Node getChildNode(int nodeIdx) throws IOException {
            int childNodeID = this.getChildNodeID(nodeIdx);
            Node childNode = BTree.this._readNode(childNodeID);
            childNode._parent = this;
            childNode._indexInParent = nodeIdx;
            return childNode;
        }

        public Node getParentNode() {
            return this._parent;
        }

        public int getIndexInParent() {
            return this._indexInParent;
        }

        public void clearParentInfo() {
            this._parent = null;
            this._indexInParent = -1;
        }

        public int search(byte[] key) {
            int low = 0;
            int high = this._valueCount - 1;
            while (low <= high) {
                int mid = low + high >> 1;
                int diff2 = BTree.this._comparator.compareBTreeValues(key, this._data, this._valueIdx2offset(mid), BTree.this._valueSize);
                if (diff2 < 0) {
                    high = mid - 1;
                    continue;
                }
                if (diff2 > 0) {
                    low = mid + 1;
                    continue;
                }
                return mid;
            }
            return -low - 1;
        }

        public void insertValueNodeIDPair(int valueIdx, byte[] value2, int nodeID) {
            int offset = this._valueIdx2offset(valueIdx);
            if (valueIdx < this._valueCount) {
                this._shiftData(offset, this._valueIdx2offset(this._valueCount), BTree.this._slotSize);
            }
            ByteArrayUtil.put((byte[])value2, (byte[])this._data, (int)offset);
            ByteArrayUtil.putInt((int)nodeID, (byte[])this._data, (int)(offset + BTree.this._valueSize));
            this._setValueCount(++this._valueCount);
            this._dataChanged = true;
        }

        public void insertNodeIDValuePair(int nodeIdx, int nodeID, byte[] value2) {
            int offset = this._nodeIdx2offset(nodeIdx);
            this._shiftData(offset, this._valueIdx2offset(this._valueCount), BTree.this._slotSize);
            ByteArrayUtil.putInt((int)nodeID, (byte[])this._data, (int)offset);
            ByteArrayUtil.put((byte[])value2, (byte[])this._data, (int)(offset + 4));
            this._setValueCount(++this._valueCount);
            this._dataChanged = true;
        }

        public byte[] splitAndInsert(byte[] newValue, int newNodeID, int newValueIdx, Node newNode) {
            this.insertValueNodeIDPair(newValueIdx, newValue, newNodeID);
            int medianIdx = BTree.this._branchFactor / 2;
            int medianOffset = this._valueIdx2offset(medianIdx);
            int splitOffset = medianOffset + BTree.this._valueSize;
            System.arraycopy(this._data, splitOffset, newNode._data, 4, this._data.length - splitOffset);
            byte[] medianValue = this.getValue(medianIdx);
            this._clearData(medianOffset, this._data.length);
            this._setValueCount(medianIdx);
            newNode._setValueCount(BTree.this._branchFactor - medianIdx - 1);
            newNode._dataChanged = true;
            return medianValue;
        }

        public void mergeWithRightSibling(byte[] medianValue, Node rightSibling) {
            this.setValue(this._valueCount, medianValue);
            System.arraycopy(rightSibling._data, 4, this._data, this._nodeIdx2offset(this._valueCount + 1), this._valueIdx2offset(rightSibling._valueCount) - 4);
            this._setValueCount(this._valueCount + 1 + rightSibling._valueCount);
            rightSibling._clearData(4, this._valueIdx2offset(rightSibling._valueCount));
            rightSibling._setValueCount(0);
            rightSibling._dataChanged = true;
        }

        public void read() throws IOException {
            ByteBuffer buf = ByteBuffer.wrap(this._data);
            buf.limit(BTree.this._nodeSize);
            BTree.this._fileChannel.read(buf, this._offset);
            this._valueCount = ByteArrayUtil.getInt((byte[])this._data, (int)0);
        }

        public void write() throws IOException {
            ByteBuffer buf = ByteBuffer.wrap(this._data);
            buf.limit(BTree.this._nodeSize);
            BTree.this._fileChannel.write(buf, this._offset);
            this._dataChanged = false;
        }

        private void _shiftData(int startOffset, int endOffset, int shift) {
            System.arraycopy(this._data, startOffset, this._data, startOffset + shift, endOffset - startOffset);
        }

        private void _clearData(int startOffset, int endOffset) {
            Arrays.fill(this._data, startOffset, endOffset, (byte)0);
        }

        private void _setValueCount(int valueCount) {
            this._valueCount = valueCount;
            ByteArrayUtil.putInt((int)this._valueCount, (byte[])this._data, (int)0);
        }

        private int _valueIdx2offset(int id) {
            return 8 + id * BTree.this._slotSize;
        }

        private int _nodeIdx2offset(int id) {
            return 4 + id * BTree.this._slotSize;
        }
    }

    private static class InsertResult {
        byte[] oldValue = null;
        byte[] overflowValue = null;
        int overflowNodeID = 0;

        private InsertResult() {
        }
    }
}

