package org.aksw.rdfunit.tests.generators;

import org.aksw.rdfunit.Utils.CacheUtils;
import org.aksw.rdfunit.Utils.TestGeneratorUtils;
import org.aksw.rdfunit.Utils.TestUtils;
import org.aksw.rdfunit.enums.TestGenerationType;
import org.aksw.rdfunit.io.reader.RDFReaderException;
import org.aksw.rdfunit.io.reader.RDFReaderFactory;
import org.aksw.rdfunit.io.reader.RDFStreamReader;
import org.aksw.rdfunit.io.writer.RDFFileWriter;
import org.aksw.rdfunit.sources.SchemaSource;
import org.aksw.rdfunit.sources.Source;
import org.aksw.rdfunit.tests.TestAutoGenerator;
import org.aksw.rdfunit.tests.TestCase;
import org.aksw.rdfunit.tests.TestSuite;
import org.aksw.rdfunit.tests.generators.monitors.TestGeneratorExecutorMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;


/**
 * <p>TestGeneratorExecutor class.</p>
 *
 * @author Dimitris Kontokostas
 *         handles test generation form a schema or a cache
 * @since 11/20/13 7:31 PM
 * @version $Id: $Id
 */
public class TestGeneratorExecutor {
    private static final Logger log = LoggerFactory.getLogger(TestGeneratorExecutor.class);
    private volatile boolean isCanceled = false;
    private final boolean loadFromCache;
    private final boolean useManualTests;
    private final boolean useAutoTests;

    /**
     * <p>Constructor for TestGeneratorExecutor.</p>
     */
    public TestGeneratorExecutor() {
        this(true, true, true);
    }

    /**
     * TestAutoGenerator constructor
     * TODO: loadFromCache does not make sense if useAutoTests is false
     *
     * @param useAutoTests a boolean.
     * @param loadFromCache a boolean.
     * @param useManualTests a boolean.
     */
    public TestGeneratorExecutor(boolean useAutoTests, boolean loadFromCache, boolean useManualTests) {
        this.useAutoTests = useAutoTests;
        this.loadFromCache = loadFromCache;
        this.useManualTests = useManualTests;

        // no auto && no manual tests do not make sense
        assert (useAutoTests || useManualTests);

        // no auto && cache does not make sense TODO fix this
        assert (useAutoTests || !loadFromCache);
    }

    private final Collection<TestGeneratorExecutorMonitor> progressMonitors = new ArrayList<>();


    /**
     * <p>cancel.</p>
     */
    public void cancel() {
        isCanceled = true;
    }


    /**
     * <p>generateTestSuite.</p>
     *
     * @param testFolder a {@link java.lang.String} object.
     * @param dataset a {@link org.aksw.rdfunit.sources.Source} object.
     * @param autoGenerators a {@link java.util.Collection} object.
     * @return a {@link org.aksw.rdfunit.tests.TestSuite} object.
     */
    public TestSuite generateTestSuite(String testFolder, Source dataset, Collection<TestAutoGenerator> autoGenerators) {

        Collection<SchemaSource> sources = dataset.getReferencesSchemata();


        /*notify start of testing */
        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.generationStarted(dataset, sources.size());
        }

        Collection<TestCase> allTests = new ArrayList<>();
        for (SchemaSource s : sources) {
            if (isCanceled) {
                break;
            }

            log.info("Generating tests for: {}", s.getUri());

            //Generate auto tests from schema
            if (useAutoTests) {
                allTests.addAll(generateAutoTestsForSchemaSource(testFolder, s, autoGenerators));
            }

            //Find manual tests for schema
            if (useManualTests) {
                allTests.addAll(generateManualTestsForSource(testFolder, s));
            }
        }

        //Find manual tests for dataset (if not canceled
        if (!isCanceled && useManualTests) {
            allTests.addAll(generateManualTestsForSource(testFolder, dataset));
        }

        /*notify start of testing */
        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.generationFinished();
        }

        return new TestSuite(allTests);
    }

    private Collection<TestCase> generateAutoTestsForSchemaSource(String testFolder, SchemaSource s, Collection<TestAutoGenerator> autoGenerators) {
        Collection<TestCase> tests = new ArrayList<>();

        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.sourceGenerationStarted(s, TestGenerationType.AutoGenerated);
        }

        try {
            String cachedTestsLocation = CacheUtils.getSourceAutoTestFile(testFolder, s);
            if (!loadFromCache) {
                cachedTestsLocation = ""; // non existing path
            }
            Collection<TestCase> testsAutoCached = TestUtils.instantiateTestsFromModel(
                    new RDFStreamReader(cachedTestsLocation).read());
            tests.addAll(testsAutoCached);
            log.info("{} contains {} automatically created tests (loaded from cache)", s.getUri(), testsAutoCached.size());

        } catch (RDFReaderException e) {
            // cannot read from file  / generate
            Collection<TestCase> testsAuto = TestGeneratorUtils.instantiateTestsFromAG(autoGenerators, s);
            tests.addAll(testsAuto);
            TestUtils.writeTestsToFile(testsAuto, new RDFFileWriter(CacheUtils.getSourceAutoTestFile(testFolder, s)));
            log.info("{} contains {} automatically created tests", s.getUri(), testsAuto.size());
        }

        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.sourceGenerationExecuted(s, TestGenerationType.AutoGenerated, tests.size());
        }

        return tests;
    }

    private Collection<TestCase> generateManualTestsForSource(String testFolder, Source s) {
        Collection<TestCase> tests = new ArrayList<>();

        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.sourceGenerationStarted(s, TestGenerationType.ManuallyGenerated);
        }
        try {
            Collection<TestCase> testsManuals = TestUtils.instantiateTestsFromModel(
                    RDFReaderFactory.createFileOrResourceReader(
                            CacheUtils.getSourceManualTestFile(testFolder, s),                 // check for local directory first
                            CacheUtils.getSourceManualTestFile("/org/aksw/rdfunit/tests/", s)  // otherwise check if it exists in resources
                    ).read());

            tests.addAll(testsManuals);
            log.info("{} contains {} manually created tests", s.getUri(), testsManuals.size());
        } catch (RDFReaderException e) {
            // Do nothing, Manual tests do not exist
        }

        for (TestGeneratorExecutorMonitor monitor : progressMonitors) {
            monitor.sourceGenerationExecuted(s, TestGenerationType.ManuallyGenerated, tests.size());
        }

        return tests;


    }


    /**
     * <p>addTestExecutorMonitor.</p>
     *
     * @param monitor a {@link org.aksw.rdfunit.tests.generators.monitors.TestGeneratorExecutorMonitor} object.
     */
    public void addTestExecutorMonitor(TestGeneratorExecutorMonitor monitor) {

        if (!progressMonitors.contains(monitor)) {
            progressMonitors.add(monitor);
        }
    }

    /**
     * <p>removeTestExecutorMonitor.</p>
     *
     * @param monitor a {@link org.aksw.rdfunit.tests.generators.monitors.TestGeneratorExecutorMonitor} object.
     */
    public void removeTestExecutorMonitor(TestGeneratorExecutorMonitor monitor) {
        progressMonitors.remove(monitor);
    }
}
