package org.aksw.limes.core.gui.model.metric;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.aksw.limes.core.gui.model.Config;
import org.aksw.limes.core.gui.util.SourceOrTarget;

/**
 * Parses Metric to Expression for the Config Reader
 *
 * @author Daniel Obraczka {@literal <} soz11ffe{@literal @}
 *         studserv.uni-leipzig.de{@literal >}
 */
public class MetricParser {
	/**
	 * Splits the String
	 *
	 * @param s
	 *            String to Split
	 * @return String list
	 */
	protected static String[] splitFunc(String s) {
		if (!s.contains("(")) {
			return new String[] { s };
		}
		final List<String> tokenList = new LinkedList<>();
		final int i = s.indexOf("(");
		tokenList.add(s.substring(0, i));
		s = s.substring(i, s.length());
		int depth = 0;
		StringBuilder sb = new StringBuilder();
		for (final char c : s.toCharArray()) {
			switch (c) {
			case '(':
				if (depth > 0) {
					sb.append(c);
				}
				depth++;
				break;
			case ')':
				if (depth > 1) {
					sb.append(c);
				}
				depth--;
				break;
			case ',':
				if (depth > 1) {
					sb.append(c);
				} else {
					tokenList.add(sb.toString());
					sb = new StringBuilder();
				}
				break;
			default:
				sb.append(c);
			}
		}
		tokenList.add(sb.toString());
		return tokenList.toArray(new String[0]);
	}

	/**
	 * Sets the parameters of a Node from StringExpression
	 *
	 * @param node
	 *            Node to be configurated
	 * @param s
	 *            Expression
	 * @param pos
	 *            Position of Node
	 * @return Node String
	 */
	protected static String setParametersFromString(Node node, String s, int pos) {
		if (node.hasFrontParameters()) {
			final int i = s.indexOf("*");
			if (i == -1) {
				return s;
			}
			final Double d = Double.valueOf(s.substring(0, i));
			s = s.substring(i + 1);
			switch (pos) {
			case 0:
				node.param1 = d;
				break;
			case 1:
				node.param2 = d;
			}
		} else {
			final int i = s.lastIndexOf("|");
			if (i == -1 || i < s.lastIndexOf(")")) {
				return s;
			}
			final Double d = Double.valueOf(s.substring(i + 1));
			s = s.substring(0, i);
			switch (pos) {
			case 0:
				node.param1 = d;
				break;
			case 1:
				node.param2 = d;
			}
		}
		return s;
	}

	/**
	 * Parses Node from String
	 *
	 * @param parent
	 *            Already parsed Parent
	 * @param s
	 *            String to parse
	 * @param sourceVar
	 *            Label
	 * @param pos
	 *            Position of Node
	 * @return Node
	 * @throws MetricFormatException
	 */
	protected static Node parsePart(Node parent, String s, String sourceVar, int pos, Config c)
			throws MetricFormatException {
		String[] tokens = splitFunc(s);
		String id = tokens[0].trim();
		if (id.contains("*")) {
			final String[] parts = id.split("\\*");
			final double d = Double.parseDouble(parts[0]);
			if (parent.param1 == null) {
				parent.param1 = d;
			} else {
				parent.param2 = d;
			}
			id = parts[1];
		}
		if (tokens.length == 1) {
			final SourceOrTarget origin = id.startsWith(sourceVar) ? SourceOrTarget.SOURCE : SourceOrTarget.TARGET;
			if (c.getSourceInfo().getOptionalProperties().contains(c.removeVar(id, origin))
					|| c.getTargetInfo().getOptionalProperties().contains(c.removeVar(id, origin))) {
				return new Property(id, origin, true);
			}
			return new Property(id, origin, false);
		}

		final Node node = Node.createNode(id);
		s = setParametersFromString(parent, s, pos);
		tokens = splitFunc(s);
		int i = 0;
		for (final String argument : Arrays.copyOfRange(tokens, 1, tokens.length)) {
			if (!node.addChild(parsePart(node, argument, sourceVar, i, c))) {
				throw new MetricFormatException("Could not add child \"" + argument + '"');
			}
			i++;
		}
		return node;
	}

	/**
	 * Parses String to Metric
	 *
	 * @param s
	 *            String to parse
	 * @param sourceVar
	 *            Label
	 * @return Metric as Output
	 * @throws MetricFormatException
	 *             thrown if there is something wrong
	 */
	public static Output parse(String s, String sourceVar, Config c) throws MetricFormatException {
		if (s.isEmpty()) {
			throw new MetricFormatException();
		}
		final Output output = new Output();
		s = setParametersFromString(output, s, 0);
		output.param1 = null;
		output.param2 = null;
		try {
			output.addChild(parsePart(output, s, sourceVar, 0, c));
		} catch (final MetricFormatException e) {
			throw new MetricFormatException("Error parsing metric expression \"" + s + "\".", e);
		}
		return output;
	}

}