/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hp.hpl.jena.graph.query;

import java.util.*;

import com.hp.hpl.jena.graph.*;
import com.hp.hpl.jena.shared.BrokenException;

/**
    A QueryTriple is a composite of three QueryNodes derived from a
    single source triple. QueryTriple knows how to classify a triple
    into a QueryTriple based on the classification of that triple's elements.
    A QueryTriple can create a specific <code>Matcher</code> which
    will do the minimal amount of work to match triples that have been
    generated by <code>find</code> on the finder of that QueryTriple.
    
    @author hedgehog
*/
public class QueryTriple
    {
    public final QueryNode S;
    public final QueryNode P;
    public final QueryNode O;
    
    public QueryTriple( QueryNode S, QueryNode P, QueryNode O )
        { this.S = S; this.P = P; this.O = O; }   
    
    @Override
    public String toString()
        { return "<qt " + S.toString() + " " + P.toString() + " " + O.toString() + ">"; }
    
    public static QueryTriple [] classify( QueryNodeFactory f, Mapping m, Triple [] t )
        {
        QueryTriple [] result = f.createArray( t.length );
        for (int i = 0; i < t.length; i += 1) result[i] = classify( f, m, t[i] );
        return result;
        }
    
    public static QueryTriple classify( QueryNodeFactory f, Mapping m, Triple t )
        { 
        HashSet<Node> fresh = new HashSet<Node>();
        return f.createTriple
            ( QueryNode.classify( f, m, fresh, t.getSubject() ), 
            QueryNode.classify( f, m, fresh, t.getPredicate() ),
            QueryNode.classify( f, m, fresh, t.getObject() ) );
        }
    
    public Applyer createApplyer( final Graph g )
        { return new SimpleApplyer( g, this ); }
    
    public Matcher createMatcher()
        {
        final int SMATCH = 4, PMATCH = 2, OMATCH = 1, NOMATCH = 0;
        int bits = 
            (S.mustMatch() ? SMATCH : 0) 
            + (P.mustMatch() ? PMATCH : 0)
            + (O.mustMatch() ? OMATCH : 0)
            ;
        switch (bits)
            {
            case SMATCH + PMATCH + OMATCH:
                return new Matcher()
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { return S.match( d, t.getSubject() )
                            && P.match( d, t.getPredicate() )
                            && O.match( d, t.getObject() ); }
                    };
                    
            case SMATCH + OMATCH:
                return new Matcher() 
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { 
                        return S.match( d, t.getSubject() ) 
                        && O.match( d, t.getObject() ); }
                    };
                    
            case SMATCH + PMATCH:  
                return new Matcher() 
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { 
                        return S.match( d, t.getSubject() ) 
                        && P.match( d, t.getPredicate() ); 
                        }
                    };
                    
            case PMATCH + OMATCH:
                return new Matcher()
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        {
                        return P.match( d, t.getPredicate() )
                        && O.match( d, t.getObject() );
                        }
                    };
    
            case SMATCH:                
                return new Matcher() 
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { return S.match( d, t.getSubject() ); }
                    };
    
            case PMATCH:
                return new Matcher()
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { return P.match( d, t.getPredicate() ); }
                    };
                    
            case OMATCH:
                return new Matcher()
                    {
                    @Override
                    public boolean match( Domain d, Triple t )
                        { return O.match( d, t.getObject() ); }
                    };
    
            case NOMATCH:
                return Matcher.always;
                    
            }
        throw new BrokenException( "uncatered-for case in optimisation" );
        }    
    
    public static class SimpleApplyer extends Applyer
        {
        protected final Graph g;
        protected final QueryNode s;
        protected final QueryNode p;
        protected final QueryNode o;

        protected SimpleApplyer( Graph g, QueryTriple qt )
            { this.g = g; this.o = qt.O; this.p = qt.P; this.s = qt.S; }

        public Iterator<Triple> find( Domain d )
            { return g.find( s.finder( d ), p.finder( d ), o.finder( d ) ); }

        @Override public void applyToTriples( Domain d, Matcher m, StageElement next )
            {
            Iterator<Triple> it = find( d );
            while (it.hasNext())
                if (m.match( d, it.next() )) 
                     next.run( d );
            }
        }
    }
