/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hp.hpl.jena.util.iterator;

import java.util.Iterator;

/** 
    boolean functions wrapped to be used in filtering iterators.
*/
public abstract class Filter<T>
    {
    /**
        Answer true iff the object <code>o</code> is acceptable. This method
        may also throw an exception if the argument is of a wrong type; it
        is not required to return <code>false</code> in such a case.
    */
	public abstract boolean accept( T o );
    
    public ExtendedIterator<T> filterKeep( Iterator<T> it )
        { return new FilterKeepIterator<>( this, it ); }
    
    public Filter<T> and( final Filter<T> other )
        { return other.isAny() ? this : new Filter<T>()
            { @Override public boolean accept( T x ) 
                { return Filter.this.accept( x ) && other.accept( x ); } 
            };
        }
    
    /**
        Answer true iff this filter will deliver true for any argument. Should never
        be overridden except by classes generated by any() below.
    */
    public boolean isAny()
        { return false; }
    
    /** 
        A Filter that accepts everything it's offered.
        @deprecated use Filter.any()
    */
    @SuppressWarnings({"unchecked", "rawtypes"})
    // Knowingly suppressed - maximum backward compatibility. 
    @Deprecated public static final Filter any = new Filter()
        { 
        @Override public final boolean isAny() { return true; }
        
        @Override public final boolean accept( Object o ) { return true; } 
        
        @Override public Filter and( Filter other ) { return other; }
        
        @Override public ExtendedIterator filterKeep( Iterator it )
            { return WrappedIterator.create( it ); }
        };
        
    public static <T> Filter<T> any() 
        {  
        return new Filter<T>()
            {
            @Override public final boolean isAny() { return true; }
            
            @Override public final boolean accept( T o ) { return true; } 
            
            @Override public Filter<T> and( Filter<T> other ) { return other; }
            
            @Override public ExtendedIterator<T> filterKeep( Iterator<T> it )
                { return WrappedIterator.create( it ); }
            };
        }
    }
