/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.testing.dbunit;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import org.dbunit.DatabaseUnitException;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.IDataSet;
import org.dbunit.operation.AbstractOperation;
import org.dbunit.operation.DatabaseOperation;
import org.dbunit.operation.ExclusiveTransactionException;


/**
 * Extended operation for dbunit that disables FK checks.
 * @author fgiust
 * @version $Id: MySqlAbstractNoFkOperation.java 578 2008-01-31 13:13:40Z fgiust $
 */
public class MySqlAbstractNoFkOperation extends AbstractOperation
{

    /**
     * Wrapped operation.
     */
    private final DatabaseOperation operation;

    /**
     * Creates a new InsertIdentityOperation object that decorates the specified operation.
     * @param operation wrapped operation
     */
    public MySqlAbstractNoFkOperation(DatabaseOperation operation)
    {
        this.operation = operation;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void execute(IDatabaseConnection connection, IDataSet dataSet) throws DatabaseUnitException, SQLException
    {
        Connection jdbcConnection = connection.getConnection();
        Statement statement = jdbcConnection.createStatement();

        try
        {
            if (!jdbcConnection.getAutoCommit())
            {
                throw new ExclusiveTransactionException();
            }
            jdbcConnection.setAutoCommit(false);

            // disable FK
            statement.execute("SET FOREIGN_KEY_CHECKS=0");

            try
            {
                operation.execute(connection, connection.createDataSet());
            }
            finally
            {
                // enable FK
                statement.execute("SET FOREIGN_KEY_CHECKS=1");
                jdbcConnection.commit();
            }

        }
        finally
        {
            jdbcConnection.setAutoCommit(true);
            statement.close();
        }
    }
}
