/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.testing;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.sql.SQLException;

import org.apache.commons.lang.StringUtils;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.FilteredDataSet;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.excel.XlsDataSet;
import org.dbunit.dataset.filter.ITableFilter;
import org.dbunit.dataset.xml.XmlDataSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fgiust
 * @version $Id: DbUnitUtils.java 581 2008-01-31 22:19:56Z fgiust $
 */
public final class DbUnitUtils
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(DbUnitUtils.class);

    /**
     * Do not instantiate.
     */
    private DbUnitUtils()
    {
        // do not instantiate
    }

    /**
     * Exports a list of table to an xml/xsl file
     * @param file The file name we save the dataset to
     * @param connection connection to use
     * @param tableNames optional list of table names
     * @throws IOException An IO Exception.
     * @throws DataSetException A dataset exception.
     * @throws SQLException A SQL Exception.
     */
    public static void exportDataset(IDatabaseConnection connection, String[] tableNames, File file)
        throws SQLException, IOException, DataSetException
    {
        OutputStream fos = null;

        try
        {
            IDataSet fullDataSet = connection.createDataSet();

            if (tableNames != null && tableNames.length > 0)
            {
                fullDataSet = new FilteredDataSet(tableNames, fullDataSet);
            }

            fos = new FileOutputStream(file);
            if (StringUtils.equalsIgnoreCase(StringUtils.substringAfterLast(file.getName(), "."), "xls"))
            {
                XlsDataSet.write(fullDataSet, fos);
            }
            else
            {
                XmlDataSet.write(fullDataSet, fos);
            }
        }
        finally
        {
            fos.close();
        }

        log.debug("Dataset exported at {}", file.getAbsolutePath());
    }

    /**
     * Exports a full db saving a dataset for each table.
     * @param connection dbunit connection
     * @param directory export directory
     * @param xls if <code>true</code> export will be done using excel datasets
     * @param tableFilter optional table filter
     * @throws SQLException sql exception
     * @throws DataSetException dbunit exception
     * @throws IOException IO exception
     */
    public void exportTablesToDir(IDatabaseConnection connection, File directory, boolean xls, ITableFilter tableFilter)
        throws SQLException, DataSetException, IOException

    {
        directory.mkdirs();

        IDataSet dataset = connection.createDataSet();
        if (tableFilter != null)
        {
            dataset = new FilteredDataSet(tableFilter, dataset);
        }

        String[] tablenames = dataset.getTableNames();

        for (String table : tablenames)
        {
            File exportFile = new File(directory, table + (xls ? ".xls" : ".xml"));
            log.info("Exporting {}", table);
            DbUnitUtils.exportDataset(connection, new String[]{table }, exportFile);
        }
    }

    /**
     * @param url dataset url (can be an xml or excel file)
     * @return loaded dataset
     * @throws IOException error reading the file
     * @throws DataSetException error loading the dataset
     */
    public static IDataSet loadDataset(URL url) throws IOException, DataSetException
    {
        if (url == null)
        {
            throw new IllegalArgumentException("URL is null");
        }
        InputStream is = url.openStream();
        IDataSet dataSet = null;

        if (is != null)
        {
            log.debug("loading dataset {}", url.getFile());
            dataSet = url.getFile().endsWith(".xls") ? new XlsDataSet(is) : new XmlDataSet(is);
            is.close();
        }
        return dataSet;
    }

}
