/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.draw.path;

import com.sencha.gxt.core.client.util.PrecisePoint;

/**
 * A {@link PathCommand} that represents a segment of an ellipse.
 */
public class EllipticalArc extends EndPointCommand {

  private double radiusX = 0;
  private double radiusY = 0;
  private double xAxisRotation = 0;
  private int largeArcFlag = 0;
  private int sweepFlag = 0;
  private String absoluteName = "A";
  private String relativeName = "a";

  /**
   * Creates an elliptical arc {@link PathCommand}. Defaults to absolute.
   */
  public EllipticalArc() {
  }

  /**
   * Creates an elliptical arc {@link PathCommand} using the given values.
   * Defaults to absolute.
   * 
   * @param radiusX the radius of the ellipse on its x-axis
   * @param radiusY the radius of the ellipse on its y-axis
   * @param xAxisRotation the rotation of the ellipse in relation to the current
   *          coordinate system
   * @param largeArcFlag if 0 then the smaller arc will be used, if 1 then the
   *          larger
   * @param sweepFlag if 0 then the arc will be in a negative angle, if 1 then
   *          in a positive angle
   * @param x the x-coordinate of the end of the segment
   * @param y the y-coordinate of the end of the segment
   */
  public EllipticalArc(double radiusX, double radiusY, double xAxisRotation, int largeArcFlag, int sweepFlag, double x,
      double y) {
    super(x, y);
    this.radiusX = radiusX;
    this.radiusY = radiusY;
    this.xAxisRotation = xAxisRotation;
    this.largeArcFlag = largeArcFlag;
    this.sweepFlag = sweepFlag;
  }

  /**
   * Creates an elliptical arc {@link PathCommand} using the given values.
   * 
   * @param radiusX the radius of the ellipse on its x-axis
   * @param radiusY the radius of the ellipse on its y-axis
   * @param xAxisRotation the rotation of the ellipse in relation to the current
   *          coordinate system
   * @param largeArcFlag if 0 then the smaller arc will be used, if 1 then the
   *          larger
   * @param sweepFlag if 0 then the arc will be in a negative angle, if 1 then
   *          in a positive angle
   * @param x the x-coordinate of the end of the segment
   * @param y the y-coordinate of the end of the segment
   * @param relative true if the command is relative
   */
  public EllipticalArc(double radiusX, double radiusY, double xAxisRotation, int largeArcFlag, int sweepFlag, double x,
      double y, boolean relative) {
    super(x, y, relative);
    this.radiusX = radiusX;
    this.radiusY = radiusY;
    this.xAxisRotation = xAxisRotation;
    this.largeArcFlag = largeArcFlag;
    this.sweepFlag = sweepFlag;
  }

  /**
   * Creates a copy of the given elliptical arc.
   * 
   * @param command the arc to be copied
   */
  public EllipticalArc(EllipticalArc command) {
    super(command);
    this.radiusX = command.radiusX;
    this.radiusY = command.radiusY;
    this.xAxisRotation = command.xAxisRotation;
    this.largeArcFlag = command.largeArcFlag;
    this.sweepFlag = command.sweepFlag;
  }

  @Override
  public EllipticalArc copy() {
    return new EllipticalArc(this);
  }

  /**
   * Returns the large arc flag of the arc.
   * 
   * @return the large arc flag of the arc
   */
  public int getLargeArcFlag() {
    return largeArcFlag;
  }

  /**
   * Returns the radius of the ellipse on its x-axis.
   * 
   * @return the radius of the ellipse on its x-axis
   */
  public double getRadiusX() {
    return radiusX;
  }

  /**
   * Returns the radius of the ellipse on its y-axis.
   * 
   * @return the radius of the ellipse on its y-axis
   */
  public double getRadiusY() {
    return radiusY;
  }

  /**
   * Returns the sweep flag of the arc.
   * 
   * @return the sweep flag of the arc
   */
  public int getSweepFlag() {
    return sweepFlag;
  }

  /**
   * Returns the rotation of the ellipse in relation to the current coordinate
   * system.
   * 
   * @return the rotation of the ellipse in relation to the current coordinate
   *         system
   */
  public double getxAxisRotation() {
    return xAxisRotation;
  }

  @Override
  public boolean nearEqual(PathCommand command) {
    if (!(command instanceof EllipticalArc)) {
      return false;
    }
    EllipticalArc arc = (EllipticalArc) command;

    if (Math.round(this.getRadiusX()) != Math.round(arc.getRadiusX())) {
      return false;
    }
    if (Math.round(this.getRadiusY()) != Math.round(arc.getRadiusY())) {
      return false;
    }
    if (Math.round(this.getxAxisRotation()) != Math.round(arc.getxAxisRotation())) {
      return false;
    }
    if (Math.round(this.getLargeArcFlag()) != Math.round(arc.getLargeArcFlag())) {
      return false;
    }
    if (Math.round(this.getSweepFlag()) != Math.round(arc.getSweepFlag())) {
      return false;
    }
    if (Math.round(this.getX()) != Math.round(arc.getX())) {
      return false;
    }
    if (Math.round(this.getY()) != Math.round(arc.getY())) {
      return false;
    }
    return true;
  }

  /**
   * Sets the large arc flag of the arc.
   * 
   * @param largeArcFlag if 0 then the smaller arc will be used, if 1 then the
   *          larger
   */
  public void setLargeArcFlag(int largeArcFlag) {
    this.largeArcFlag = largeArcFlag;
  }

  /**
   * Sets the radius of the ellipse on its x-axis.
   * 
   * @param radiusX the radius of the ellipse on its x-axis
   */
  public void setRadiusX(double radiusX) {
    this.radiusX = radiusX;
  }

  /**
   * Sets the radius of the ellipse on its y-axis.
   * 
   * @param radiusY the radius of the ellipse on its y-axis
   */
  public void setRadiusY(double radiusY) {
    this.radiusY = radiusY;
  }

  /**
   * Sets the sweep flag of the arc.
   * 
   * @param sweepFlag if 0 then the arc will be in a negative angle, if 1 then
   *          in a positive angle
   */
  public void setSweepFlag(int sweepFlag) {
    this.sweepFlag = sweepFlag;
  }

  /**
   * Sets the rotation of the ellipse in relation to the current coordinate
   * system
   * 
   * @param xAxisRotation the rotation of the ellipse in relation to the current
   *          coordinate system
   */
  public void setxAxisRotation(double xAxisRotation) {
    this.xAxisRotation = xAxisRotation;
  }

  @Override
  public CurveTo toCurve(PrecisePoint currentPoint, PrecisePoint movePoint, PrecisePoint curvePoint,
      PrecisePoint quadraticPoint) {
    quadraticPoint.setX(currentPoint.getX());
    quadraticPoint.setY(currentPoint.getY());
    return arc2curve(currentPoint, this);
  }

  @Override
  public String toString() {
    StringBuilder build = new StringBuilder();
    if (!relative) {
      build.append(absoluteName);
    } else {
      build.append(relativeName);
    }
    build.append(radiusX).append(",").append(radiusY).append(",").append(xAxisRotation).append(",").append(largeArcFlag).append(
        ",").append(sweepFlag).append(",").append(x).append(",").append(y);
    return build.toString();
  }

  /**
   * Converts an {@link EllipticalArc} to {@link CurveTo}.
   * 
   * @param currentPoint the current point on the path
   * @param arc the elliptical arc to be converted
   * @return the resultant {@link CurveTo} command
   */
  private CurveTo arc2curve(PrecisePoint currentPoint, EllipticalArc arc) {
    // for more information of where this Math came from visit:
    // http://www.w3.org/TR/SVG11/implnote.html#ArcImplementationNotes
    CurveTo curveto = new CurveTo();
    PrecisePoint radii = new PrecisePoint(arc.getRadiusX(), arc.getRadiusY());
    PrecisePoint end = new PrecisePoint(arc.getX(), arc.getY());
    PrecisePoint f = null;
    PrecisePoint c = null;
    double rad = Math.toRadians(arc.getxAxisRotation());

    currentPoint = rotatePoint(currentPoint, -rad);
    end = rotatePoint(end, -rad);

    PrecisePoint point = new PrecisePoint((currentPoint.getX() - end.getX()) / 2.0,
        (currentPoint.getY() - end.getY()) / 2.0);

    double h = (point.getX() * point.getX()) / (radii.getX() * radii.getX()) + (point.getY() * point.getY())
        / (radii.getY() * radii.getY());
    if (h > 1) {
      h = Math.sqrt(h);
      radii.setX(h * radii.getX());
      radii.setY(h * radii.getY());
    }
    PrecisePoint rsquared = new PrecisePoint(radii.getX() * radii.getX(), radii.getY() * radii.getY());
    double k = Math.sqrt(Math.abs((rsquared.getX() * rsquared.getY() - rsquared.getX() * point.getY() * point.getY() - rsquared.getY()
        * point.getX() * point.getX())
        / (rsquared.getX() * point.getY() * point.getY() + rsquared.getY() * point.getX() * point.getX())));
    k = (arc.getLargeArcFlag() == arc.getSweepFlag() ? -1 : 1) * k;

    c = new PrecisePoint(k * radii.getX() * point.getY() / radii.getY() + (currentPoint.getX() + end.getX()) / 2, k
        * -radii.getY() * point.getX() / radii.getX() + (currentPoint.getY() + end.getY()) / 2);
    f = new PrecisePoint(Math.asin((currentPoint.getY() - c.getY()) / radii.getY()), Math.asin((end.getY() - c.getY())
        / radii.getY()));

    if (currentPoint.getX() < c.getX()) {
      f.setX(Math.PI - f.getX());
    }
    if (end.getX() < c.getX()) {
      f.setY(Math.PI - f.getY());
    }
    if (f.getX() < 0) {
      f.setX(Math.PI * 2.0 + f.getX());
    }
    if (f.getY() < 0) {
      f.setY(Math.PI * 2.0 + f.getY());
    }
    if (arc.getSweepFlag() != 0 && f.getX() > f.getY()) {
      f.setX(f.getX() - Math.PI * 2.0);
    }
    if (arc.getSweepFlag() == 0 && f.getY() > f.getX()) {
      f.setY(f.getY() - Math.PI * 2.0);
    }

    double tan = Math.tan((f.getY() - f.getX()) / 4.0);
    PrecisePoint hpoint = new PrecisePoint(4.0 / 3.0 * radii.getX() * tan, 4.0 / 3.0 * radii.getY() * tan);

    PrecisePoint m2 = new PrecisePoint(currentPoint.getX() + hpoint.getX() * Math.sin(f.getX()), currentPoint.getY()
        - hpoint.getY() * Math.cos(f.getX()));
    PrecisePoint m3 = new PrecisePoint(end.getX() + hpoint.getX() * Math.sin(f.getY()), end.getY() - hpoint.getY()
        * Math.cos(f.getY()));
    PrecisePoint m4 = new PrecisePoint(end);
    m2.setX(2.0 * currentPoint.getX() - m2.getX());
    m2.setY(2.0 * currentPoint.getY() - m2.getY());

    m2 = rotatePoint(m2, rad);
    curveto.setX1(m2.getX());
    curveto.setY1(m2.getY());

    m3 = rotatePoint(m3, rad);
    curveto.setX2(m3.getX());
    curveto.setY2(m3.getY());

    m4 = rotatePoint(m4, rad);
    curveto.setX(m4.getX());
    curveto.setY(m4.getY());

    return curveto;
  }

  /**
   * Rotates a point by the passed degrees.
   * 
   * @param point the point to be rotated
   * @param degrees the amount to rotate the point
   * @return the rotated point
   */
  private PrecisePoint rotatePoint(PrecisePoint point, double degrees) {
    double cos = Math.cos(degrees);
    double sin = Math.sin(degrees);
    double x = point.getX();
    double y = point.getY();

    return new PrecisePoint(x * cos - y * sin, x * sin + y * cos);
  }

}
