/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.draw.engine;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style.Unit;
import com.sencha.gxt.chart.client.draw.Gradient;
import com.sencha.gxt.chart.client.draw.Matrix;
import com.sencha.gxt.chart.client.draw.Rotation;
import com.sencha.gxt.chart.client.draw.Scaling;
import com.sencha.gxt.chart.client.draw.Stop;
import com.sencha.gxt.chart.client.draw.Surface;
import com.sencha.gxt.chart.client.draw.Translation;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.CircleSprite;
import com.sencha.gxt.chart.client.draw.sprite.EllipseSprite;
import com.sencha.gxt.chart.client.draw.sprite.ImageSprite;
import com.sencha.gxt.chart.client.draw.sprite.RectangleSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite.TextAnchor;
import com.sencha.gxt.core.client.GXT;
import com.sencha.gxt.core.client.dom.XDOM;
import com.sencha.gxt.core.client.dom.XElement;
import com.sencha.gxt.core.client.util.PreciseRectangle;

/**
 * Provides specific methods to draw with SVG.
 */
public class SVG extends Surface {
  /**
   * {@link JavaScriptObject} representing bounding box results of an
   * {@link SVG} text element.
   */
  public static final class TextBBox extends JavaScriptObject {

    protected TextBBox() {
    }

    /**
     * Returns the height of the SVG text element
     * 
     * @return the height of the SVG text element
     */
    public native double getHeight() /*-{
                                     return this.height;
                                     }-*/;

    /**
     * Returns the width of the SVG text element
     * 
     * @return the width of the SVG text element
     */
    public native double getWidth() /*-{
                                    return this.width;
                                    }-*/;

    /**
     * Returns the x-coordinate of the SVG text element
     * 
     * @return the x-coordinate of the SVG text element
     */
    public native double getX() /*-{
                                return this.x;
                                }-*/;

    /**
     * Returns the y-coordinate of the SVG text element
     * 
     * @return the y-coordinate of the SVG text element
     */
    public native double getY() /*-{
                                return this.y;
                                }-*/;

  }

  private static int clipId = 0;
  private Element defs;
  private Map<Sprite, XElement> elements = new HashMap<Sprite, XElement>();
  private Map<Sprite, XElement> clipElements = new HashMap<Sprite, XElement>();

  @Override
  public void addGradient(Gradient... gradients) {
    for (Gradient gradient : gradients) {
      this.gradients.add(gradient);
      Element gradientElement;
      double radAngle = Math.toRadians(gradient.getAngle());
      double[] vector = {0, 0, Math.cos(radAngle), Math.sin(radAngle)};
      double temp = Math.max(Math.abs(vector[2]), Math.abs(vector[3]));
      if (temp == 0) temp = 1;
      double max = 1 / temp;
      vector[2] *= max;
      vector[3] *= max;
      if (vector[2] < 0) {
        vector[0] = -vector[2];
        vector[2] = 0;
      }
      if (vector[3] < 0) {
        vector[1] = -vector[3];
        vector[3] = 0;
      }
      gradientElement = this.createSVGElement("linearGradient");
      gradientElement.setAttribute("x1", String.valueOf(vector[0]));
      gradientElement.setAttribute("y1", String.valueOf(vector[1]));
      gradientElement.setAttribute("x2", String.valueOf(vector[2]));
      gradientElement.setAttribute("y2", String.valueOf(vector[3]));
      gradientElement.setId(gradient.getId());
      getDefinitions().appendChild(gradientElement);
      for (int i = 0; i < gradient.getStops().size(); i++) {
        Stop stop = gradient.getStops().get(i);
        Element stopEl = createSVGElement("stop");
        stopEl.setAttribute("offset", String.valueOf(stop.getOffset()) + "%");
        stopEl.setAttribute("stop-color", stop.getColor().toString());
        stopEl.setAttribute("stop-opacity", String.valueOf(stop.getOpacity()));
        gradientElement.appendChild(stopEl);
      }
    }
  }

  @Override
  public void draw() {
    super.draw();
    if (surfaceElement == null) {
      surfaceElement = this.createSVGElement("svg");
      surfaceElement.setAttribute("xmlns", "http://www.w3.org/2000/svg");
      surfaceElement.setAttribute("version", "1.1");
      surfaceElement.setAttribute("width", String.valueOf(width));
      surfaceElement.setAttribute("height", String.valueOf(height));
      Element bgRect = this.createSVGElement("rect");
      bgRect.setAttribute("width", "100%");
      bgRect.setAttribute("height", "100%");
      bgRect.setAttribute("fill", "#000");
      bgRect.setAttribute("stroke", "none");
      bgRect.setAttribute("opacity", "0");

      surfaceElement.appendChild(getDefinitions());
      surfaceElement.appendChild(bgRect);
      container.appendChild(surfaceElement);
    }

    Collections.sort(sprites, zIndexComparator());

    renderAll();
  }

  @Override
  public void renderSprite(Sprite sprite) {
    if (surfaceElement == null) {
      return;
    }
    if (getElement(sprite) == null) {
      createSprite(sprite);
    }
    if (!sprite.isDirty()) {
      return;
    }
    if (sprite.isZIndexDirty()) {
      applyZIndex(sprite);
    }

    applyAttributes(sprite);
    if (sprite.isTransformDirty()) {
      transform(sprite);
    }
    sprite.clearDirtyFlags();
  }

  @Override
  public void setCursor(Sprite sprite, String property) {
    XElement element = getElement(sprite);
    if (element != null) {
      element.getStyle().setProperty("cursor", property);
    }
  }

  @Override
  public void setViewBox(double x, double y, double width, double height) {
    surfaceElement.setAttribute(
        "viewBox",
        new StringBuilder().append(x).append(", ").append(y).append(", ").append(width).append(", ").append(height).toString());
  }

  @Override
  protected void deleteSprite(Sprite sprite) {
    surfaceElement.removeChild(getElement(sprite));
  }

  @Override
  protected PreciseRectangle getBBoxText(TextSprite sprite) {
    PreciseRectangle bbox = new PreciseRectangle();
    XElement element = getElement(sprite);

    if (element != null) {
      TextBBox box = getTextBBox(element);
      bbox.setX(box.getX());
      bbox.setY(box.getY());
      bbox.setWidth(box.getWidth());
      bbox.setHeight(box.getHeight());
    }

    return bbox;
  }

  /**
   * Applies the attributes of the given sprite to its SVG element.
   * 
   * @param sprite the sprite to have its attributes set
   */
  private void applyAttributes(Sprite sprite) {
    XElement element = getElement(sprite);

    if (sprite instanceof PathSprite) {
      PathSprite path = (PathSprite) sprite;
      if (path.isPathDirty()) {
        if (path.size() > 0) {
          element.setAttribute("d", path.toString());
        } else {
          element.removeAttribute("d");
        }
      }
      if (path.isStrokeLineCapDirty()) {
        setAttribute(element, "stroke-linecap", path.getStrokeLineCap());
      }
      if (path.isStrokeLineJoinDirty()) {
        setAttribute(element, "stroke-linejoin", path.getStrokeLineJoin());
      }
      if (path.isMiterLimitDirty()) {
        setAttribute(element, "stroke-miterlimit", path.getMiterLimit());
      }
    } else if (sprite instanceof TextSprite) {
      TextSprite text = (TextSprite) sprite;
      if (text.isTextDirty() || text.isXDirty()) {
        tuneText(text);
      }
      if (text.isFontSizeDirty()) {
        if (text.getFontSize() > 0) {
          element.getStyle().setFontSize(text.getFontSize(), Unit.PX);
        } else {
          element.getStyle().clearFontSize();
        }
      }
      if (text.isFontStyleDirty()) {
        if (text.getFontStyle() != null) {
          element.getStyle().setFontStyle(text.getFontStyle());
        } else {
          element.getStyle().clearFontStyle();
        }
      }
      if (text.isFontWeightDirty()) {
        if (text.getFontWeight() != null) {
          element.getStyle().setFontWeight(text.getFontWeight());
        } else {
          element.getStyle().clearFontWeight();
        }
      }
      if (text.isFontDirty()) {
        setAttribute(element, "font-family", text.getFont());
      }
      if (text.isTextAnchorDirty()) {
        if (text.getTextAnchor() == TextAnchor.START) {
          element.setAttribute("text-anchor", "start");
        } else if (text.getTextAnchor() == TextAnchor.MIDDLE) {
          element.setAttribute("text-anchor", "middle");
        } else if (text.getTextAnchor() == TextAnchor.END) {
          element.setAttribute("text-anchor", "end");
        } else {
          element.removeAttribute("text-anchor");
        }
      }
      if (text.isXDirty()) {
        setAttribute(element, "x", text.getX());
      }
      if (text.isYDirty()) {
        setAttribute(element, "y", text.getY());
      }
    } else if (sprite instanceof RectangleSprite) {
      RectangleSprite rect = (RectangleSprite) sprite;
      if (rect.isXDirty()) {
        setAttribute(element, "x", rect.getX());
      }
      if (rect.isYDirty()) {
        setAttribute(element, "y", rect.getY());
      }
      if (rect.isWidthDirty()) {
        setAttribute(element, "width", rect.getWidth());
      }
      if (rect.isWidthDirty()) {
        setAttribute(element, "height", rect.getHeight());
      }
      if (rect.isRadiusDirty()) {
        setAttribute(element, "rx", rect.getRadius());
        setAttribute(element, "ry", rect.getRadius());
      }
    } else if (sprite instanceof CircleSprite) {
      CircleSprite circle = (CircleSprite) sprite;
      if (circle.isCenterXDirty()) {
        setAttribute(element, "cx", circle.getCenterX());
      }
      if (circle.isCenterYDirty()) {
        setAttribute(element, "cy", circle.getCenterY());
      }
      if (circle.isRadiusDirty()) {
        setAttribute(element, "r", circle.getRadius());
      }
    } else if (sprite instanceof EllipseSprite) {
      EllipseSprite ellipse = (EllipseSprite) sprite;
      if (ellipse.isCenterXDirty()) {
        setAttribute(element, "cx", ellipse.getCenterX());
      }
      if (ellipse.isCenterYDirty()) {
        setAttribute(element, "cy", ellipse.getCenterY());
      }
      if (ellipse.isRadiusXDirty()) {
        setAttribute(element, "rx", ellipse.getRadiusX());
      }
      if (ellipse.isRadiusYDirty()) {
        setAttribute(element, "ry", ellipse.getRadiusY());
      }
    } else if (sprite instanceof ImageSprite) {
      ImageSprite image = (ImageSprite) sprite;
      if (image.isResourceDirty() && image.getResource() != null) {
        element.setAttributeNS("http://www.w3.org/1999/xlink", "href", image.getResource().getSafeUri().asString());
      }
      if (image.isXDirty()) {
        setAttribute(element, "x", image.getX());
      }
      if (image.isYDirty()) {
        setAttribute(element, "y", image.getY());
      }
      if (image.isHeightDirty() && !Double.isNaN(image.getHeight())) {
        element.setAttribute("height", image.getHeight() + "px");
      }
      if (image.isWidthDirty() && !Double.isNaN(image.getWidth())) {
        element.setAttribute("width", image.getWidth() + "px");
      }
    }

    if (sprite.isStrokeDirty()) {
      setAttribute(element, "stroke", sprite.getStroke());
    }
    if (sprite.isStrokeWidthDirty()) {
      setAttribute(element, "stroke-width", sprite.getStrokeWidth());
    }
    if (sprite.isFillDirty()) {
      setAttribute(element, "fill", sprite.getFill());
    }
    if (sprite.isFillOpacityDirty()) {
      setAttribute(element, "fill-opacity", sprite.getFillOpacity());
    }
    if (sprite.isStrokeOpacityDirty()) {
      setAttribute(element, "stroke-opacity", sprite.getStrokeOpacity());
    }
    if (sprite.isOpacityDirty()) {
      setAttribute(element, "opacity", sprite.getOpacity());
    }

    // Hide or show the sprite
    if (sprite.isHiddenDirty()) {
      if (sprite.isHidden()) {
        element.setAttribute("visibility", "hidden");
      } else {
        element.removeAttribute("visibility");
      }
    }

    // Apply clip rectangle to the sprite
    if (sprite.isClipRectangleDirty()) {
      if (sprite.getClipRectangle() != null) {
        applyClip(sprite);
      }
    }
  }

  /**
   * Applies the clipping rectangle of the given sprite
   * 
   * @param sprite the sprite to apply its clipping rectangle
   */
  private void applyClip(Sprite sprite) {
    PreciseRectangle rect = sprite.getClipRectangle();
    XElement clipPath = getClipElement(sprite);

    if (clipPath != null) {
      Element clipParent = clipPath.getParentElement();
      clipParent.getParentElement().removeChild(clipParent);
    }

    Element clipElement = createSVGElement("clipPath");
    clipPath = createSVGElement("rect");
    clipElement.setId("ext-clip-" + ++clipId);
    clipPath.setAttribute("x", String.valueOf(rect.getX()));
    clipPath.setAttribute("y", String.valueOf(rect.getY()));
    clipPath.setAttribute("width", String.valueOf(rect.getWidth()));
    clipPath.setAttribute("height", String.valueOf(rect.getHeight()));
    clipElement.appendChild(clipPath);
    getDefinitions().appendChild(clipElement);
    getElement(sprite).setAttribute("clip-path", "url(#" + clipElement.getId() + ")");
    setClipElement(sprite, clipPath);
  }

  /**
   * Insert or move a given {@link Sprite}'s element to the correct place in the
   * DOM list for its zIndex.
   * 
   * @param sprite - the {@link Sprite} to insert into the dom
   */
  private void applyZIndex(Sprite sprite) {
    int index = insertSprite(sprite);
    Element element = getElement(sprite);
    Element previousElement = null;

    if (index + 2 >= this.surfaceElement.getChildCount() || !this.surfaceElement.getChild(index + 2).equals(element)) {
      // shift by 2 to account for defs and bg rect
      if (index > 0) {
        // Find the first previous sprite which has its DOM element created
        // already
        while (previousElement == null && index > 0) {
          previousElement = getElement(sprites.get(--index));
        }
      }
      this.surfaceElement.insertAfter(element, previousElement);
    }
  }

  /**
   * Generates an SVG element for the given sprite and inserts it into the DOM
   * based on its z-index.
   * 
   * @param sprite the sprite to have its element generated
   * @return the generated element
   */
  private Element createSprite(Sprite sprite) {
    // Create svg element and append to the DOM.
    final XElement element;
    if (sprite instanceof CircleSprite) {
      element = createSVGElement("circle");
    } else if (sprite instanceof EllipseSprite) {
      element = createSVGElement("ellipse");
    } else if (sprite instanceof ImageSprite) {
      element = createSVGElement("image");
    } else if (sprite instanceof PathSprite) {
      element = createSVGElement("path");
    } else if (sprite instanceof RectangleSprite) {
      element = createSVGElement("rect");
    } else if (sprite instanceof TextSprite) {
      element = createSVGElement("text");
    } else {
      element = null;
    }
    setElement(sprite, element);
    element.getStyle().setProperty("webkitTapHighlightColor", "rgba(0,0,0,0)");
    applyZIndex(sprite); // performs the insertion
    return element;
  }

  /**
   * Creates an SVG element using the give type.
   * 
   * @param type the type of element to create
   * @return the created SVG element
   */
  private XElement createSVGElement(String type) {
    return XElement.as(XDOM.createElementNS("http://www.w3.org/2000/svg", type));
  }

  /**
   * Returns the clip element associated with the given sprite.
   * 
   * @param sprite the sprite
   * @return the clip element
   */
  private XElement getClipElement(Sprite sprite) {
    return clipElements.get(sprite);
  }

  /**
   * Returns the definitions element of the SVG element. If not already created
   * it will be instantiated.
   * 
   * @return the definitions element of the SVG element
   */
  private Element getDefinitions() {
    if (defs == null) {
      defs = createSVGElement("defs");
    }
    return defs;
  }

  /**
   * Returns the DOM element associated with the given sprite.
   * 
   * @param sprite the sprite
   * @return the DOM element
   */
  private XElement getElement(Sprite sprite) {
    return elements.get(sprite);
  }

  /**
   * Returns the {@link TextBBox} for the given text element.
   * 
   * @param text the text element to get the bounding box
   * @return the bounding box
   */
  private native TextBBox getTextBBox(Element text) /*-{
                                                    return text.getBBox();
                                                    }-*/;

  /**
   * Returns the height of an SVG Text Element.
   * 
   * @param text the element to have its height calculated
   * @return the height of an SVG Text Element
   */
  private native double getTextHeight(Element text) /*-{
                                                    return text.getBBox().height;
                                                    }-*/;

  /**
   * Returns the width of an SVG Text Element.
   * 
   * @param text the element to have its width calculated
   * @return the width of an SVG Text Element
   */
  private native double getTextWidth(Element text) /*-{
                                                   return text.getBBox().width;
                                                   }-*/;

  /**
   * Returns the x coordinate of an SVG Text Element.
   * 
   * @param text the element to have its x coordinate calculated
   * @return the x coordinate of an SVG Text Element
   */
  private native double getTextX(Element text) /*-{
                                               return text.getBBox().x;
                                               }-*/;

  /**
   * Returns the y coordinate of an SVG Text Element.
   * 
   * @param text the element to have its y coordinate calculated
   * @return the y coordinate of an SVG Text Element
   */
  private native double getTextY(Element text) /*-{
                                               return text.getBBox().y;
                                               }-*/;

  /**
   * Insert or move a given {@link Sprite} into the correct position in the
   * items {@link SpriteList}, according to its zIndex. Will be inserted at the
   * end of an existing series of sprites with the same or lower zIndex. If the
   * sprite is already positioned within an appropriate zIndex group, it will
   * not be moved. This ordering can be used by subclasses to assist in
   * rendering the sprites in the correct order for proper z-index stacking.
   * 
   * @param sprite the sprite to be inserted
   * @return the sprite's new index in the list
   */
  private int insertSprite(Sprite sprite) {
    int zIndex = sprite.getzIndex();
    int index = sprites.indexOf(sprite);

    if (index < 0 || (index > 0 && sprites.get(index - 1).getzIndex() > zIndex)
        || (index < sprites.size() - 1 && sprites.get(index + 1).getzIndex() < zIndex)) {
      if (index >= 0) {
        sprites.remove(index);
      }

      index = Collections.binarySearch(sprites, sprite, zIndexComparator());
      if (index < 0) {
        index = sprites.size();
      }
      sprites.add(index, sprite);
    }

    return index;
  }

  private void setAttribute(XElement element, String attribute, double value) {
    if (!(Double.isNaN(value))) {
      element.setAttribute(attribute, String.valueOf(value));
    } else {
      element.removeAttribute(attribute);
    }
  }

  private void setAttribute(XElement element, String attribute, Object value) {
    if (value != null) {
      element.setAttribute(attribute, value.toString());
    } else {
      element.removeAttribute(attribute);
    }
  }

  /**
   * Associates the given sprite with the given clip element.
   * 
   * @param sprite the sprite
   * @param element the clip element
   */
  private void setClipElement(Sprite sprite, XElement element) {
    clipElements.put(sprite, element);
  }

  /**
   * Associates the given sprite with the given dom element.
   * 
   * @param sprite the sprite
   * @param element the dom element
   */
  private void setElement(Sprite sprite, XElement element) {
    elements.put(sprite, element);
  }

  /**
   * Divides up the text of the given {@link TextSprite}. Tspans are created
   * based on the new line character. The resultant tspans are added to the text
   * element.
   * 
   * @param sprite the sprite to have its text used
   * @return the generated tspans
   */
  private List<Element> setText(TextSprite sprite) {
    List<Element> tspans = new ArrayList<Element>();
    XElement spriteElement = getElement(sprite);

    while (spriteElement.hasChildNodes()) {
      spriteElement.removeChild(spriteElement.getFirstChild());
    }

    // Wrap each row into tspan to emulate rows
    String[] texts = sprite.getText().split("\n");
    for (int i = 0; i < texts.length; i++) {
      if (texts[i] != null) {
        Element tspan = createSVGElement("tspan");
        tspan.appendChild(XDOM.createTextNode(texts[i]));
        tspan.setAttribute("x", String.valueOf(sprite.getX()));
        spriteElement.appendChild(tspan);
        tspans.add(tspan);
      }
    }

    return tspans;
  }

  /**
   * Applies the transformations of the given sprite to its SVG element. The
   * transformations applied include {@link Rotation}, {@link Scaling} and
   * {@link Translation}.
   * 
   * @param sprite the sprite to have its transformations applied
   */
  private void transform(Sprite sprite) {
    Matrix matrix = sprite.transformMatrix();
    getElement(sprite).setAttribute(
        "transform",
        new StringBuilder().append("matrix( ").append(matrix.get(0, 0)).append(", ").append(matrix.get(1, 0)).append(
            ", ").append(matrix.get(0, 1)).append(", ").append(matrix.get(1, 1)).append(", ").append(matrix.get(0, 2)).append(
            ", ").append(matrix.get(1, 2)).append(")").toString());
  }

  /**
   * Wrap SVG text inside a tspan to allow for line wrapping. In addition this
   * normalizes the baseline for text the vertical middle of the text to be the
   * same as VML.
   * 
   * @param sprite - the sprite to have its text tuned
   */
  private void tuneText(TextSprite sprite) {
    double lineHeight = 1.2;
    List<Element> tspans = new ArrayList<Element>();

    if (sprite.getText() != null) {
      tspans = setText(sprite);
    }

    // Normalize baseline via a DY shift of first tspan. Shift other rows by
    // height * line height (1.2)
    if (tspans.size() > 0) {
      double height = getTextHeight(getElement(sprite));
      for (int i = 0; i < tspans.size(); i++) {
        // The text baseline for FireFox 3.0 and 3.5 is different than other SVG
        // implementations
        // so we are going to normalize that here
        double factor = GXT.isGecko1_9() ? 2 : 4;
        tspans.get(i).setAttribute("dy", String.valueOf(i != 0 ? height * lineHeight : height / factor));
      }
    }
  }

  /**
   * Generates a {@link Comparator} for use in sorting sprites by their z-index.
   * 
   * @return the generated comparator
   */
  private Comparator<Sprite> zIndexComparator() {
    return new Comparator<Sprite>() {
      @Override
      public int compare(Sprite o1, Sprite o2) {
        return o1.getzIndex() - o2.getzIndex();
      }
    };
  }
}
