/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.event.shared.HandlerRegistration;
import com.sencha.gxt.chart.client.chart.Chart;
import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.event.SeriesHandler;
import com.sencha.gxt.chart.client.chart.event.SeriesHandler.HasSeriesHandlers;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOutEvent;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOutEvent.HasSeriesItemOutHandlers;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOutEvent.SeriesItemOutHandler;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOverEvent;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOverEvent.HasSeriesItemOverHandlers;
import com.sencha.gxt.chart.client.chart.event.SeriesItemOverEvent.SeriesItemOverHandler;
import com.sencha.gxt.chart.client.chart.event.SeriesSelectionEvent;
import com.sencha.gxt.chart.client.chart.event.SeriesSelectionEvent.HasSeriesSelectionHandlers;
import com.sencha.gxt.chart.client.chart.event.SeriesSelectionEvent.SeriesSelectionHandler;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.path.EndPointCommand;
import com.sencha.gxt.chart.client.draw.path.PathCommand;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.core.shared.event.GroupingHandlerRegistration;
import com.sencha.gxt.data.shared.LabelProvider;
import com.sencha.gxt.widget.core.client.tips.ToolTip;

/**
 * Series is the abstract class containing the common logic to all chart series.
 * This class implements the logic of handling mouse events, animating, hiding,
 * showing all elements and returning the color of the series to be used as a
 * legend item.
 * 
 * @param <M> the data type of the series
 */
public abstract class Series<M> implements HasSeriesHandlers, HasSeriesSelectionHandlers, HasSeriesItemOutHandlers,
    HasSeriesItemOverHandlers {

  /**
   * Enumeration used for labels positioned on the series.
   */
  public enum LabelPosition {
    /**
     * Label positioned at the start of the series.
     */
    START,
    /**
     * Label positioned at the end of the series.
     */
    END,
    /**
     * Label positioned outside of the series.
     */
    OUTSIDE
  }

  protected Chart<M> chart;
  protected Position yAxisPosition;
  protected ValueProvider<M, ? extends Number> xField;
  protected SpriteList<Sprite> sprites = new SpriteList<Sprite>();
  protected List<SpriteList<Sprite>> shadowGroups = new ArrayList<SpriteList<Sprite>>();
  protected List<Sprite> shadowAttributes = new ArrayList<Sprite>();
  protected PreciseRectangle bbox = new PreciseRectangle();
  protected boolean shownInLegend = true;
  protected Map<Integer, String> legendNames = new HashMap<Integer, String>();
  protected boolean highlighting = false;
  protected int lastHighlighted = -1;
  protected SeriesHighlighter highlighter;
  protected SeriesRenderer<M> renderer;
  protected ToolTip toolTip;
  protected SeriesToolTipConfig<M> toolTipConfig;
  protected Map<Integer, Sprite> labels = new HashMap<Integer, Sprite>();
  protected SeriesLabelConfig<M> labelConfig;
  private HandlerManager handlerManager;
  protected List<String> legendTitles = new ArrayList<String>();
  protected Color stroke;
  protected double strokeWidth = 0;

  /**
   * Creates a series.
   */
  public Series() {
  }

  public HandlerRegistration addSeriesHandler(SeriesHandler handler) {
    GroupingHandlerRegistration reg = new GroupingHandlerRegistration();
    reg.add(ensureHandlers().addHandler(SeriesSelectionEvent.getType(), handler));
    reg.add(ensureHandlers().addHandler(SeriesItemOutEvent.getType(), handler));
    reg.add(ensureHandlers().addHandler(SeriesItemOverEvent.getType(), handler));
    chart.sinkBrowserEvents();
    return reg;
  }

  @Override
  public HandlerRegistration addSeriesItemOutHandler(SeriesItemOutHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(SeriesItemOutEvent.getType(), handler);
  }

  @Override
  public HandlerRegistration addSeriesItemOverHandler(SeriesItemOverHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(SeriesItemOverEvent.getType(), handler);
  }

  @Override
  public HandlerRegistration addSeriesSelectionHandler(SeriesSelectionHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(SeriesSelectionEvent.getType(), handler);
  }

  /**
   * Calculates the bounding the box of the series and stores the result. To get
   * the result us {@link #getBBox()}.
   * 
   * @param gutter true if to use the series gutter in the calculation
   */
  public void calculateBBox(boolean gutter) {
    PreciseRectangle chartBBox = chart.getBBox();
    PreciseRectangle zoom = chart.getZoom();
    double gutterX = 0;
    double gutterY = 0;
    if (gutter) {
      gutterX = chart.getMaxGutter()[0];
      gutterY = chart.getMaxGutter()[1];
    }

    bbox = new PreciseRectangle();
    bbox.setX((chartBBox.getX() + gutterX) - (zoom.getX() * zoom.getWidth()));
    bbox.setY((chartBBox.getY() + gutterY) - (zoom.getY() * zoom.getHeight()));
    bbox.setWidth((chartBBox.getWidth() - (gutterX * 2)) * zoom.getWidth());
    bbox.setHeight((chartBBox.getHeight() - (gutterY * 2)) * zoom.getHeight());
  }

  /**
   * Removes all the sprites of the series from the surface.
   */
  public void delete() {
    while (sprites.size() > 0) {
      sprites.remove(sprites.size() - 1).remove();
    }
    for (Integer integer : labels.keySet()) {
      Sprite sprite = labels.get(integer);
      if (sprite != null) {
        labels.put(integer, null);
        sprite.remove();
      }
    }
  }

  /**
   * Draws the series for the current chart.
   */
  public abstract void drawSeries();

  /**
   * Returns the bounding box of the series.
   * 
   * @return the bounding box of the series
   */
  public PreciseRectangle getBBox() {
    return bbox;
  }

  /**
   * Returns the chart that the series is attached.
   * 
   * @return the chart that the series is attached
   */
  public Chart<M> getChart() {
    return chart;
  }

  /**
   * Returns the axis insets of the series.
   * 
   * @return the axis insets of the series
   */
  public double[] getGutters() {
    double[] temp = {0, 0};
    return temp;
  }

  /**
   * Return the {@link SeriesHighlighter} used by the series.
   * 
   * @return the series highlighter
   */
  public SeriesHighlighter getHighlighter() {
    return highlighter;
  }

  /**
   * Returns the configuration for labels on the series.
   * 
   * @return the configuration for labels on the series
   */
  public SeriesLabelConfig<M> getLabelConfig() {
    return labelConfig;
  }

  /**
   * Returns the map of names used in the legend.
   * 
   * @return the map of names used in the legend
   */
  public Map<Integer, String> getLegendNames() {
    return legendNames;
  }

  /**
   * Returns the list of titles used in the legend of the series.
   * 
   * @return the list of titles used in the legend of the series
   */
  public List<String> getLegendTitles() {
    return legendTitles;
  }

  /**
   * Return the custom sprite renderer on the series.
   * 
   * @return the custom sprite renderer on the series
   */
  public SeriesRenderer<M> getRenderer() {
    return renderer;
  }

  /**
   * Return the stroke color of the series.
   * 
   * @return the stroke color of the series
   */
  public Color getStroke() {
    return stroke;
  }

  /**
   * Return the stroke width of the series.
   * 
   * @return the stroke width of the series
   */
  public double getStrokeWidth() {
    return strokeWidth;
  }

  /**
   * Return the generated tool tip from the tool tip config.
   * 
   * @return the generated tool tip
   */
  public ToolTip getToolTip() {
    if (toolTip == null && toolTipConfig != null) {
      toolTip = new ToolTip(null, toolTipConfig);
    }
    return toolTip;
  }

  /**
   * Returns the tooltip configuration.
   * 
   * @return the tooltip configuration
   */
  public SeriesToolTipConfig<M> getToolTipConfig() {
    return toolTipConfig;
  }

  /**
   * Returns the value provider for the x-axis of the series.
   * 
   * @return the value provider for the x-axis of the series
   */
  public ValueProvider<M, ? extends Number> getXField() {
    return xField;
  }

  /**
   * Returns the y axis position of the series.
   * 
   * @return the y axis position of the series
   */
  public Position getYAxisPosition() {
    return yAxisPosition;
  }

  /**
   * Hides the given y field index from the series.
   * 
   * @param yFieldIndex the index of the y field
   */
  public abstract void hide(int yFieldIndex);

  /**
   * Highlights the series at the given index.
   * 
   * @param index the index to be highlighted
   */
  public abstract void highlight(int index);

  /**
   * Highlights all of the items in the series.
   * 
   * @param index the index of the series
   */
  public abstract void highlightAll(int index);

  /**
   * Returns whether or not the series is actively highlighted.
   * 
   * @return true if highlighted
   */
  public boolean highlightedState() {
    if (lastHighlighted != -1) {
      return true;
    }
    return false;
  }

  /**
   * Returns whether or not the series uses highlighting.
   * 
   * @return whether or not the series uses highlighting
   */
  public boolean isHighlighting() {
    return highlighting;
  }

  /**
   * Returns whether or not the series is shown in the legend.
   * 
   * @return true if the series is shown in the legend
   */
  public boolean isShownInLegend() {
    return shownInLegend;
  }

  /**
   * Method used when the series is clicked.
   * 
   * @param point the point clicked
   */
  public void onMouseDown(PrecisePoint point) {
    int index = getIndex(point);
    if (index > -1) {
      ensureHandlers().fireEvent(new SeriesSelectionEvent(getValue(index), index));
    }
  }

  /**
   * Method used when the series is moused over.
   * 
   * @param point the point moused over
   * @return the index of the moused over item
   */
  public int onMouseMove(PrecisePoint point) {
    int index = getIndex(point);
    if (lastHighlighted > -1 && index != lastHighlighted) {
      onMouseOut(point);
    }
    if (index > -1) {
      ensureHandlers().fireEvent(new SeriesItemOverEvent(getValue(index), index));
      if (toolTip != null) {
        LabelProvider<Number> numeric = toolTipConfig.getNumericLabelProvider();
        LabelProvider<M> custom = toolTipConfig.getCustomLabelProvider();
        if (custom != null) {
          toolTipConfig.setBodyText(custom.getLabel(chart.getCurrentStore().get(index)));
        } else if (numeric != null) {
          toolTipConfig.setBodyText(numeric.getLabel(getValue(index).doubleValue()));
        }
        toolTip.update(toolTipConfig);
        toolTip.showAt((int) Math.round(point.getX() + chart.getAbsoluteLeft() + 20),
            (int) Math.round(point.getY() + chart.getAbsoluteTop() + 20));
      }
      if (highlighting) {
        highlight(index);
      }
      lastHighlighted = index;
    }

    return index;
  }

  /**
   * Method used when the mouse leaves the series.
   * 
   * @param point the point left
   */
  public void onMouseOut(PrecisePoint point) {
    if (toolTip != null) {
      toolTip.hide();
    }
    if (lastHighlighted != -1) {
      ensureHandlers().fireEvent(new SeriesItemOutEvent(getValue(lastHighlighted), lastHighlighted));
      if (highlighting) {
        unHighlight(lastHighlighted);
      }
      lastHighlighted = -1;
    }
  }

  /**
   * Removes the components tooltip (if one exists).
   */
  public void removeToolTip() {
    if (toolTip != null) {
      toolTip.initTarget(null);
      toolTip = null;
      toolTipConfig = null;
    }
  }

  /**
   * Sets the chart that the series is attached.
   * 
   * @param chart the chart that the series is attached
   */
  public void setChart(Chart<M> chart) {
    this.chart = chart;
    if (chart != null && highlighting || toolTipConfig != null) {
      chart.sinkBrowserEvents();
    }
  }

  /**
   * Set the {@link SeriesHighlighter} used by the series.
   * 
   * @param highlighter the series highlighter
   */
  public void setHighlighter(SeriesHighlighter highlighter) {
    this.highlighter = highlighter;
  }

  /**
   * Sets whether or not the series uses highlighting.
   * 
   * @param highlighting whether or not the series uses highlighting
   */
  public void setHighlighting(boolean highlighting) {
    this.highlighting = highlighting;
    if (highlighting && chart != null) {
      chart.sinkBrowserEvents();
    }
  }

  /**
   * Sets the configuration for labels on the series.
   * 
   * @param labelConfig the label configuration
   */
  public void setLabelConfig(SeriesLabelConfig<M> labelConfig) {
    if (this.labelConfig != labelConfig) {
      this.labelConfig = labelConfig;
      int size = labels.size();
      for (int i = 0; i < size; i++) {
        labels.remove(i).remove();
      }
    }
  }

  /**
   * Sets a custom sprite renderer on the series.
   * 
   * @param renderer the renderer
   */
  public void setRenderer(SeriesRenderer<M> renderer) {
    this.renderer = renderer;
  }

  /**
   * Sets whether or not the series is shown in the legend.
   * 
   * @param showInLegend true if the series is shown in the legend
   */
  public void setShownInLegend(boolean showInLegend) {
    this.shownInLegend = showInLegend;
  }

  /**
   * Sets the stroke color of the series.
   * 
   * @param stroke the stroke color of the series
   */
  public void setStroke(Color stroke) {
    this.stroke = stroke;
  }

  /**
   * Sets the stroke width of the series.
   * 
   * @param strokeWidth the stroke width of the series
   */
  public void setStrokeWidth(double strokeWidth) {
    this.strokeWidth = strokeWidth;
  }

  /**
   * Sets the tooltip configuration.
   * 
   * @param config the tooltip configuration
   */
  public void setToolTipConfig(SeriesToolTipConfig<M> config) {
    this.toolTipConfig = config;
    if (config != null) {
      if (chart != null) {
        chart.sinkBrowserEvents();
      }
      if (toolTip == null) {
        toolTip = new ToolTip(null, config);
      } else {
        toolTip.update(config);
      }
    } else if (config == null) {
      removeToolTip();
    }
  }

  /**
   * Sets the value provider for the x-axis of the series.
   * 
   * @param xField the value provider for the x-axis of the series
   */
  public void setXField(ValueProvider<M, ? extends Number> xField) {
    this.xField = xField;
  }

  /**
   * Sets the position of the y axis on the chart to be used by the series.
   * 
   * @param yAxisPosition the position of the y axis on the chart to be used by
   *          the series
   */
  public void setYAxisPosition(Position yAxisPosition) {
    this.yAxisPosition = yAxisPosition;
  }

  /**
   * Shows the given y field index from the series.
   * 
   * @param yFieldIndex the index of the y field
   */
  public abstract void show(int yFieldIndex);

  /**
   * Removes highlighting from the given index.
   * 
   * @param index the index to have its highlighting removed
   */
  public abstract void unHighlight(int index);

  /**
   * UnHighlights all items in the series.
   * 
   * @param index the index of the series
   */
  public abstract void unHighlightAll(int index);

  /**
   * Returns whether or not the given index is visible in legend.
   * 
   * @param index the index to determine visible
   * @return whether or not it is visible
   */
  public abstract boolean visibleInLegend(int index);

  protected HandlerManager ensureHandlers() {
    if (handlerManager == null) {
      handlerManager = new HandlerManager(this);
    }
    return handlerManager;
  }

  /**
   * Returns the index from the given point.
   * 
   * @param point the point get the index
   * @return the index
   */
  protected abstract int getIndex(PrecisePoint point);

  /**
   * Returns the end point of the given command.
   * 
   * @param command the command to get the point from
   * @return the end point of the command
   */
  protected PrecisePoint getPointFromCommand(PathCommand command) {
    if (command instanceof EndPointCommand) {
      EndPointCommand end = (EndPointCommand) command;
      return new PrecisePoint(end.getX(), end.getY());
    } else {
      return new PrecisePoint();
    }
  }

  /**
   * Returns the value at the given index.
   * 
   * @param index the index
   * @return the value
   */
  protected abstract Number getValue(int index);

  /**
   * Attempts to get a simplified yet meaningful string from the given
   * {@link ValueProvider}.
   * 
   * @param provider the value provider
   * @return the name
   */
  protected String getValueProviderName(ValueProvider<M, ? extends Number> provider) {
    String name = provider.getClass().getName();
    int start = name.indexOf('_') + 1;
    name = name.substring(start, name.indexOf('_', start));
    return name;
  }

  /**
   * Generates label text for the given sprite at the given index.
   * 
   * @param sprite the sprite to be set
   * @param index the index of the label data
   */
  protected void setLabelText(Sprite sprite, int index) {
    if (sprite instanceof TextSprite) {
      TextSprite text = (TextSprite) sprite;
      LabelProvider<Number> numeric = labelConfig.getNumericLabelProvider();
      LabelProvider<M> custom = labelConfig.getCustomLabelProvider();
      if (numeric != null) {
        text.setText(numeric.getLabel((getValue(index))));
      } else if (custom != null) {
        text.setText(custom.getLabel(chart.getCurrentStore().get(index)));
      }
      text.redraw();
    }
  }

}
