/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.List;

import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.Legend;
import com.sencha.gxt.chart.client.chart.axis.Axis;
import com.sencha.gxt.chart.client.chart.axis.NumericAxis;
import com.sencha.gxt.chart.client.draw.DrawFx;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.Translation;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.CircleSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.data.shared.ListStore;

/**
 * Creates a Scatter Chart. The scatter plot is useful when trying to display
 * more than two variables in the same visualization. These variables can be
 * mapped into x, y coordinates and also to an element's radius/size, color,
 * etc.
 * 
 * @param <M> the data type used by this series
 */
public class ScatterSeries<M> extends Series<M> {

  // The display style for the scatter series markers.
  protected Sprite markerConfig;
  protected double selectionTolerance = 20;

  protected ValueProvider<M, ? extends Number> yField;
  private String title;

  private List<PrecisePoint> coordinates = new ArrayList<PrecisePoint>();

  /**
   * Creates a scatter {@link Series}.
   */
  public ScatterSeries() {
    // setup shadow attributes
    Sprite config = new PathSprite();
    config.setStrokeWidth(6);
    config.setStrokeOpacity(0.05);
    config.setStroke(RGB.BLACK);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(4);
    config.setStrokeOpacity(0.1);
    config.setStroke(RGB.BLACK);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(2);
    config.setStrokeOpacity(0.15);
    config.setStroke(RGB.BLACK);
    shadowAttributes.add(config);

    setHighlighter(new ScatterHighlighter());

    legendTitles.add("");

    CircleSprite circle = new CircleSprite();
    circle.setRadius(8);
    markerConfig = circle;
  }

  @Override
  public void drawSeries() {
    PreciseRectangle chartBBox = chart.getBBox();
    ListStore<M> store = chart.getCurrentStore();
    double minX = Double.NaN;
    double maxX = Double.NaN;
    double minY = Double.NaN;
    double maxY = Double.NaN;
    double xScale;
    double yScale;

    // if the store is empty then there's nothing to be rendered
    if (store == null || store.size() == 0) {
      return;
    }

    // initialize the shadow groups
    if (chart.hasShadows() && shadowGroups.size() == 0) {
      for (int i = 0; i < shadowAttributes.size(); i++) {
        shadowGroups.add(new SpriteList<Sprite>());
      }
    }

    bbox.setX(chartBBox.getX() + chart.getMaxGutter()[0]);
    bbox.setY(chartBBox.getY() + chart.getMaxGutter()[1]);
    bbox.setWidth(chartBBox.getWidth() - (chart.getMaxGutter()[0] * 2));
    bbox.setHeight(chartBBox.getHeight() - (chart.getMaxGutter()[1] * 2));

    // clipRect = chartBBox;

    if (yAxisPosition != null) {
      Axis<M, ?> currentAxis = chart.getAxis(yAxisPosition);
      if (currentAxis != null) {
        if (currentAxis.getPosition() == Position.TOP || currentAxis.getPosition() == Position.BOTTOM) {
          minX = currentAxis.getFrom();
          maxX = currentAxis.getTo();
        } else {
          minY = currentAxis.getFrom();
          maxY = currentAxis.getTo();
        }
      }
    }
    // If a field was specified without a corresponding axis, create one to get
    // bounds
    if (xField != null && Double.isNaN(minX)) {
      NumericAxis<M> axis = new NumericAxis<M>();
      axis.setChart(chart);
      axis.addField(xField);
      axis.calcEnds();
      minX = axis.getFrom();
      maxX = axis.getTo();
    }
    if (yField != null && Double.isNaN(minY)) {
      NumericAxis<M> axis = new NumericAxis<M>();
      axis.setChart(chart);
      axis.addField(yField);
      axis.calcEnds();
      minY = axis.getFrom();
      maxY = axis.getTo();
    }

    xScale = bbox.getWidth() / (maxX - minX);
    yScale = bbox.getHeight() / (maxY - minY);

    coordinates.clear();

    for (int i = 0; i < store.size(); i++) {
      final double xValue, x, y;
      double yValue = yField.getValue(store.get(i)).doubleValue();
      // Ensure a value
      if (xField == null) {
        xValue = i;
        minX = 0;
        xScale = bbox.getWidth() / (store.size() - 1);
      } else {
        xValue = xField.getValue(store.get(i)).doubleValue();
      }
      x = bbox.getX() + (xValue - minX) * xScale;
      y = bbox.getY() + bbox.getHeight() - (yValue - minY) * yScale;
      coordinates.add(new PrecisePoint(x, y));
    }
    // Create new or reuse sprites and animate/display
    for (int i = 0; i < chart.getStore().size(); i++) {
      final Sprite sprite;
      if (i < sprites.size()) {
        sprite = sprites.get(i);
      } else {
        sprite = markerConfig.copy();
        sprite.setTranslation((bbox.getX() + bbox.getWidth()) / 2, (bbox.getY() + bbox.getHeight()) / 2);
        sprites.add(sprite);

        if (chart.hasShadows()) {
          for (int shindex = 0; shindex < shadowGroups.size(); shindex++) {
            PathSprite shadowSprite = new PathSprite();
            shadowSprite.setTranslation((bbox.getX() + bbox.getWidth()) / 2, (bbox.getY() + bbox.getHeight()) / 2);
            if (renderer != null) {
              renderer.spriteRenderer(shadowSprite, i, store);
            }
          }
        }
        chart.addSprite(sprite);
      }
      double x = coordinates.get(i).getX();
      double y = coordinates.get(i).getY();
      if (chart.isAnimated()) {
        DrawFx.createTranslationAnimator(sprite, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
      } else {
        sprite.setTranslation(x, y);
      }
      if (renderer != null) {
        renderer.spriteRenderer(sprite, i, store);
      }

      sprite.redraw();
    }
    drawLabels();
  }

  /**
   * Returns the marker configuration.
   * 
   * @return the marker configuration
   */
  public Sprite getMarkerConfig() {
    return markerConfig;
  }

  /**
   * Returns the selection tolerance of markers.
   * 
   * @return the selection tolerance of markers
   */
  public double getSelectionTolerance() {
    return selectionTolerance;
  }

  /**
   * Returns the series title used in the legend.
   * 
   * @return the series title used in the legend
   */
  public String getTitle() {
    return title;
  }

  /**
   * Returns the {@link ValueProvider} used for the y axis of the series.
   * 
   * @return the value provider used for the y axis of the series
   */
  public ValueProvider<M, ? extends Number> getYField() {
    return yField;
  }

  @Override
  public void hide(int yFieldIndex) {
    toggle(true);
  }

  @Override
  public void highlight(int yFieldIndex) {
    Sprite sprite = sprites.get(yFieldIndex);
    if (highlighter != null) {
      highlighter.hightlight(sprite);
    }
  }

  @Override
  public void highlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.hightlight(sprites.get(i));
    }
  }

  /**
   * Set the series title used in the legend.
   * 
   * @param title the series title used in the legend
   */
  public void setLegendTitle(String title) {
    if (title != null) {
      legendTitles.set(0, title);
    } else {
      legendTitles.set(0, getValueProviderName(yField));
    }
    this.title = title;
    if (chart != null) {
      Legend<M> legend = chart.getLegend();
      if (legend != null) {
        legend.create();
        legend.updatePosition();
      }
    }
  }

  /**
   * Sets the marker configuration.
   * 
   * @param markerConfig the marker configuration
   */
  public void setMarkerConfig(Sprite markerConfig) {
    if (this.markerConfig != markerConfig) {
      this.markerConfig = markerConfig;
      delete();
    }
  }

  /**
   * Sets the selection tolerance of markers.
   * 
   * @param selectionTolerance the selection tolerance of markers
   */
  public void setSelectionTolerance(double selectionTolerance) {
    this.selectionTolerance = selectionTolerance;
  }

  /**
   * Sets the {@link ValueProvider} used for the y axis of the series.
   * 
   * @param yField the value provider
   */
  public void setYField(ValueProvider<M, ? extends Number> yField) {
    this.yField = yField;
    if (title == null) {
      legendTitles.set(0, getValueProviderName(yField));
    }
  }

  @Override
  public void show(int yFieldIndex) {
    toggle(false);
  }

  @Override
  public void unHighlight(int yFieldIndex) {
    if (highlighter != null) {
      highlighter.unHightlight(sprites.get(yFieldIndex));
    }
  }

  @Override
  public void unHighlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.unHightlight(sprites.get(i));
    }
  }

  @Override
  public boolean visibleInLegend(int index) {
    if (sprites.size() == 0) {
      return true;
    } else {
      return !sprites.get(0).isHidden();
    }
  }

  @Override
  protected int getIndex(PrecisePoint point) {
    for (int i = 0; i < sprites.size(); i++) {
      Translation trans = sprites.get(i).getTranslation();
      if (point.equalsNoPrecision(new PrecisePoint(trans.getX(), trans.getY()), selectionTolerance)) {
        return i;
      }
    }
    return -1;
  }

  @Override
  protected Number getValue(int index) {
    return yField.getValue((chart.getCurrentStore().get(index)));
  }

  /**
   * Draws the labels on the series.
   */
  private void drawLabels() {
    if (labelConfig != null) {
      for (int i = 0; i < chart.getStore().size(); i++) {
        final Sprite sprite;
        if (labels.get(i) != null) {
          sprite = labels.get(i);
        } else {
          sprite = labelConfig.getSpriteConfig().copy();
          labels.put(i, sprite);
          chart.addSprite(sprite);
        }
        setLabelText(sprite, i);
        PreciseRectangle spriteBox = sprites.get(i).getBBox();
        PreciseRectangle textBox = sprite.getBBox();
        double x = coordinates.get(i).getX();
        double y = coordinates.get(i).getY();

        LabelPosition position = labelConfig.getLabelPosition();
        if (position == LabelPosition.START) {
          y -= textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
        } else if (position == LabelPosition.OUTSIDE) {
          y += textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
        }

        if (chart.isAnimated() && sprite.getTranslation() != null) {
          DrawFx.createTranslationAnimator(sprite, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
        } else {
          sprite.setTranslation(x, y);
          sprite.redraw();
        }
      }
    }
  }

  /**
   * Toggles all the sprites in the series to be hidden or shown.
   * 
   * @param hide if true hides
   */
  private void toggle(boolean hide) {
    if (sprites.size() > 0) {
      for (int i = 0; i < sprites.size(); i++) {
        sprites.get(i).setHidden(hide);
        sprites.get(i).redraw();
      }
    }
  }
}
