/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.List;

import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.DrawFx;
import com.sencha.gxt.chart.client.draw.path.ClosePath;
import com.sencha.gxt.chart.client.draw.path.LineTo;
import com.sencha.gxt.chart.client.draw.path.MoveTo;
import com.sencha.gxt.chart.client.draw.path.PathCommand;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.data.shared.ListStore;

/**
 * Creates a Radar Chart. A Radar Chart is a useful visualization technique for
 * comparing different quantitative values for a constrained number of
 * categories.
 * 
 * @param <M> the data type used by this series
 */
public class RadarSeries<M> extends ScatterSeries<M> {

  private double centerX;
  private double centerY;
  private double radius;
  private List<ValueProvider<M, ? extends Number>> fields = new ArrayList<ValueProvider<M, ? extends Number>>();
  private PathSprite radar;
  private boolean showMarkers = false;
  private Color fill = Color.NONE;
  private List<PrecisePoint> items = new ArrayList<PrecisePoint>();
  private SeriesRenderer<M> lineRenderer;

  /**
   * Creates a radar series.
   */
  public RadarSeries() {
    setHighlighter(new ScatterHighlighter());
    strokeWidth = 0.5;
  }

  @Override
  public void delete() {
    super.delete();
    if (radar != null) {
      radar.remove();
      radar = null;
    }
  }

  @Override
  public void drawSeries() {
    PreciseRectangle bbox = chart.getBBox();
    ListStore<M> store = chart.getCurrentStore();
    double maxValue = 0;
    double rho = 0;
    double x = 0;
    double y = 0;
    double len = store.size();
    List<PathCommand> commands = new ArrayList<PathCommand>();
    items.clear();

    centerX = bbox.getX() + (bbox.getWidth() / 2);
    centerY = bbox.getY() + (bbox.getHeight() / 2);
    radius = Math.min(bbox.getWidth(), bbox.getHeight()) / 2;

    // get all renderer fields
    for (Series<M> series : chart.getSeries()) {
      if (series instanceof RadarSeries) {
        final RadarSeries<M> radar = (RadarSeries<M>) series;
        fields.add(radar.getYField());
      }
    }

    // get maxValue to interpolate
    for (int index = 0; index < store.size(); index++) {
      for (int i = 0; i < fields.size(); i++) {
        maxValue = Math.max(fields.get(i).getValue(store.get(index)).doubleValue(), maxValue);
      }
    }

    // create path and items
    for (int i = 0; i < len; i++) {
      M model = store.get(i);
      rho = radius * yField.getValue(model).doubleValue() / maxValue;
      x = rho * Math.cos(i / len * Math.PI * 2);
      y = rho * Math.sin(i / len * Math.PI * 2);
      if (i == 0) {
        commands.add(new MoveTo(x + centerX, y + centerY));
      } else {
        commands.add(new LineTo(x + centerX, y + centerY));
      }
      items.add(new PrecisePoint(x + centerX, y + centerY));
    }
    commands.add(new ClosePath());

    // create path sprite
    if (radar == null) {
      radar = new PathSprite();
      chart.addSprite(radar);
    }
    if (!Double.isNaN(strokeWidth)) {
      radar.setStrokeWidth(strokeWidth);
    }
    if (fill != null) {
      radar.setFill(fill);
    }
    if (stroke != null) {
      radar.setStroke(stroke);
    }
    if (chart.isAnimated() && radar.size() > 0) {
      DrawFx.createCommandsAnimator(radar, commands).run(500);
    } else {
      radar.setCommands(commands);
      radar.redraw();
    }
    if (lineRenderer != null) {
      lineRenderer.spriteRenderer(radar, 0, chart.getStore());
    }
    if (showMarkers) {
      drawMarkers();
    }
    drawLabels();
  }

  /**
   * Returns the fill of the line in the series.
   * 
   * @return the fill of the line in the series
   */
  public Color getFill() {
    return fill;
  }

  /**
   * Returns the {@link SeriesRenderer} used on the line sprite.
   * 
   * @return the series renderer used on the line sprite
   */
  public SeriesRenderer<M> getLineRenderer() {
    return lineRenderer;
  }

  @Override
  public void hide(int yFieldIndex) {
    toggle(true);
  }

  @Override
  public void highlight(int yFieldIndex) {
    highlighter.hightlight(sprites.get(yFieldIndex));
  }

  @Override
  public void highlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.hightlight(sprites.get(i));
    }
  }

  /**
   * Returns whether or not the series shows markers.
   * 
   * @return true if markers shown
   */
  public boolean isShowMarkers() {
    return showMarkers;
  }

  /**
   * Sets the fill of the line in the series.
   * 
   * @param fill the fill of the line in the series
   */
  public void setFill(Color fill) {
    this.fill = fill;
  }

  /**
   * Sets the {@link SeriesRenderer} used on the line sprite
   * 
   * @param lineRenderer the series renderer used on the line sprite
   */
  public void setLineRenderer(SeriesRenderer<M> lineRenderer) {
    this.lineRenderer = lineRenderer;
  }

  /**
   * Sets whether or not the series shows markers.
   * 
   * @param showMarkers true if markers shown
   */
  public void setShowMarkers(boolean showMarkers) {
    if (this.showMarkers != showMarkers) {
      this.showMarkers = showMarkers;
      sprites.clear();
    }
  }

  @Override
  public void show(int yFieldIndex) {
    toggle(false);
  }

  @Override
  public void unHighlight(int yFieldIndex) {
    highlighter.unHightlight(sprites.get(yFieldIndex));
  }

  @Override
  public void unHighlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.unHightlight(sprites.get(i));
    }
  }

  @Override
  public boolean visibleInLegend(int index) {
    if (radar == null) {
      return true;
    } else {
      return !radar.isHidden();
    }
  }

  @Override
  protected int getIndex(PrecisePoint point) {
    for (int i = 0; i < radar.size() - 1; i++) {
      PathCommand command = radar.getCommand(i);
      PrecisePoint bound = getPointFromCommand(command);
      if (point.equalsNoPrecision(bound, selectionTolerance)) {
        return i;
      }
    }
    return -1;
  }

  @Override
  protected Number getValue(int index) {
    return yField.getValue((chart.getCurrentStore().get(index)));
  }

  /**
   * Draws the labels on the series.
   */
  private void drawLabels() {
    if (labelConfig != null) {
      for (int i = 0; i < items.size(); i++) {
        final Sprite sprite;
        if (labels.get(i) != null) {
          sprite = labels.get(i);
        } else {
          sprite = labelConfig.getSpriteConfig().copy();
          labels.put(i, sprite);
          chart.addSprite(sprite);
        }
        setLabelText(sprite, i);
        PrecisePoint point = items.get(i);
        double x = point.getX();
        double y = point.getY();
        if (showMarkers) {
          PreciseRectangle spriteBox = sprites.get(i).getBBox();
          PreciseRectangle textBox = sprite.getBBox();

          LabelPosition labelPosition = labelConfig.getLabelPosition();
          if (labelPosition == LabelPosition.START) {
            y -= textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
          } else if (labelPosition == LabelPosition.OUTSIDE) {
            y += textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
          }
        }
        if (chart.isAnimated() && sprite.getTranslation() != null) {
          DrawFx.createTranslationAnimator(sprite, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
        } else {
          sprite.setTranslation(x, y);
          sprite.redraw();
        }
      }
    }
  }

  /**
   * Renders markers to the surface of the series.
   */
  private void drawMarkers() {
    for (int i = 0; i < items.size(); i++) {
      PrecisePoint item = items.get(i);
      final Sprite marker;
      if (i < sprites.size()) {
        marker = sprites.get(i);
      } else {
        marker = markerConfig.copy();
        chart.addSprite(marker);
        sprites.add(marker);
      }
      if (chart.isAnimated() && marker.getTranslation() != null) {
        DrawFx.createTranslationAnimator(marker, item.getX(), item.getY()).run(chart.getAnimationDuration(),
            chart.getAnimationEasing());
      } else {
        marker.setTranslation(item.getX(), item.getY());
        marker.redraw();
      }
      if (renderer != null) {
        renderer.spriteRenderer(marker, i, chart.getCurrentStore());
      }
    }
  }

  /**
   * Toggles all the sprites in the series to be hidden or shown.
   * 
   * @param hide if true hides
   */
  private void toggle(boolean hide) {
    if (radar != null) {
      radar.setHidden(hide);
      radar.redraw();
    }
    if (sprites.size() > 0) {
      for (int i = 0; i < sprites.size(); i++) {
        sprites.get(i).setHidden(hide);
        sprites.get(i).redraw();
      }
    }
  }

}
