/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.List;

import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.axis.Axis;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.DrawFx;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.Translation;
import com.sencha.gxt.chart.client.draw.path.LineTo;
import com.sencha.gxt.chart.client.draw.path.MoveTo;
import com.sencha.gxt.chart.client.draw.path.PathCommand;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.data.shared.ListStore;

/**
 * Creates a Line Chart. A Line Chart is a useful visualization technique to
 * display quantitative information for different categories or other real
 * values (as opposed to the {@link BarSeries}), that can show some progression
 * (or regression) in the data set.
 * 
 * @param <M> the data type used by the series
 */
public class LineSeries<M> extends ScatterSeries<M> {

  private PathSprite line;
  private PathSprite fillSprite;
  private SpriteList<PathSprite> lineShadows = new SpriteList<PathSprite>();
  private boolean showMarkers = false;
  private boolean smooth = false;
  private int segments = 6;
  private Color fill = RGB.NONE;
  private List<Double> xValues = new ArrayList<Double>();
  private List<Double> yValues = new ArrayList<Double>();
  private List<PathCommand> lineCommands = new ArrayList<PathCommand>();
  private List<PathCommand> previousCommands;
  private double markerIndex = 0;
  private SeriesRenderer<M> lineRenderer;
  private SeriesRenderer<M> fillRenderer;

  /**
   * Creates a line {@link Series}.
   */
  public LineSeries() {
    // setup shadow attributes
    shadowAttributes = new ArrayList<Sprite>();
    Sprite config = new PathSprite();
    config.setStrokeWidth(6);
    config.setStrokeOpacity(0.05);
    config.setStroke(RGB.BLACK);
    config.setTranslation(1, 1);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(4);
    config.setStrokeOpacity(0.1);
    config.setStroke(RGB.BLACK);
    config.setTranslation(1, 1);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(2);
    config.setStrokeOpacity(0.15);
    config.setStroke(RGB.BLACK);
    config.setTranslation(1, 1);
    shadowAttributes.add(config);

    setHighlighter(new ScatterHighlighter());
    strokeWidth = 0.5;
  }

  @Override
  public void delete() {
    super.delete();
    if (line != null) {
      line.remove();
      line = null;
    }
    if (fillSprite != null) {
      fillSprite.remove();
      fillSprite = null;
    }
    if (lineShadows != null) {
      lineShadows.clear();
    }
  }

  @Override
  public void drawSeries() {
    String dummyPath = "M";
    List<PathCommand> fillCommands = new ArrayList<PathCommand>();
    double[] gutter = chart.getMaxGutter();
    Axis<M, ?> axis;
    boolean onbreak = false;
    double minX = Double.NaN;
    double maxX = 0;
    double minY = Double.NaN;
    double maxY = 0;
    double xScale = 0;
    double yScale = 0;
    double firstY = Double.NaN;
    Sprite marker;
    PreciseRectangle bbox = new PreciseRectangle();
    PreciseRectangle chartBBox = chart.getBBox();
    ListStore<M> store = chart.getCurrentStore();
    xValues.clear();
    yValues.clear();
    lineCommands.clear();

    // initialize shadow groups
    if (chart.hasShadows() && shadowGroups.size() == 0) {
      for (int i = 0; i < shadowAttributes.size(); i++) {
        shadowGroups.add(new SpriteList<Sprite>());
      }
    }

    if (markerIndex > 0 && sprites.size() > 0) {
      for (int i = 0; i < markerIndex; i++) {
        marker = sprites.get(i);
        sprites.remove(i);
        sprites.add(marker);
        Sprite markerTemp = sprites.get(sprites.size() - 2);
        marker.setTranslation(new Translation(markerTemp.getTranslation()));
        marker.redraw();
      }
    }

    bbox.setX((int) (chartBBox.getX() + gutter[0]));
    bbox.setY((int) (chartBBox.getY() + gutter[1]));
    bbox.setWidth((int) (chartBBox.getWidth() - (gutter[0] * 2)));
    bbox.setHeight((int) (chartBBox.getHeight() - (gutter[1] * 2)));
    axis = chart.getAxis(yAxisPosition);
    if (axis != null) {
      if (axis.getPosition() == Position.TOP || axis.getPosition() == Position.BOTTOM) {
        minX = axis.getFrom();
        maxX = axis.getTo();
      } else {
        minY = axis.getFrom();
        maxY = axis.getTo();
      }
    }

    if (Double.isNaN(minX)) {
      minX = 0;
      xScale = bbox.getWidth() / (store.size() - 1);
    } else {
      xScale = bbox.getWidth() / (maxX - minX);
    }
    if (Double.isNaN(minY)) {
      minY = 0;
      yScale = bbox.getHeight() / (store.size() - 1);
    } else {
      yScale = bbox.getHeight() / (maxY - minY);
    }

    for (int i = 0; i < store.size(); i++) {
      M model = store.get(i);
      double xValue = 0;
      // Ensure a value
      if (xField == null) {
        xValue = i;
      } else {
        xValue = xField.getValue(model).doubleValue();
      }

      xValues.add(xValue);
      yValues.add(yField.getValue(model).doubleValue());
    }

    double x = 0;
    for (int i = 0; i < xValues.size(); i++) {
      double xValue = xValues.get(i);
      double yValue = yValues.get(i);

      double y = Double.NaN;
      if (Double.isNaN(yValue)) {
        lineCommands.clear();
        onbreak = true;
        continue;
      } else {
        x = Math.round((bbox.getX() + (xValue - minX) * xScale) * 10) / 10;
        y = Math.round(((bbox.getY() + bbox.getHeight()) - (yValue - minY) * yScale) * 10) / 10;
        if (onbreak) {
          onbreak = false;
        }
        if (lineCommands.size() > 0) {
          lineCommands.add(new LineTo(x, y));
        } else {
          lineCommands.add(new MoveTo(x, y));
        }

      }

      if (Double.isNaN(firstY) && !Double.isNaN(y)) {
        firstY = y;
      }
      // If this is the first line, create a dummy path to animate in from.
      if (line == null || chart.isResizing()) {
        dummyPath += x + " " + (bbox.getY() + bbox.getHeight() / 2) + " ";
      }

      if (showMarkers) {
        if (i < sprites.size()) {
          marker = sprites.get(i);
        } else {
          marker = markerConfig.copy();
          sprites.add(marker);
          chart.addSprite(marker);
        }
        if (chart.isAnimated() && marker.getTranslation() != null
            && ((markerIndex <= 0) || (markerIndex > 0 && i != sprites.size() - 1))) {
          DrawFx.createTranslationAnimator(marker, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
        } else {
          marker.setTranslation(x, y);
          marker.redraw();
        }
        if (renderer != null) {
          renderer.spriteRenderer(marker, i, store);
        }
      }
    }

    List<PathCommand> renderCommands;

    if (smooth) {
      PathSprite smooth = new PathSprite();
      smooth.setCommands(lineCommands);
      renderCommands = smooth.copy().toSmooth(segments).getCommands();
    } else {
      renderCommands = PathSprite.copyCommands(lineCommands);
    }

    // Correct path if we're animating timeAxis intervals
    if (markerIndex > 0 && previousCommands != null && previousCommands.size() > 1) {
      previousCommands.remove(1);
      line.setCommands(previousCommands);
      if (chart.hasShadows()) {
        for (int i = 0; i < lineShadows.size(); i++) {
          PathSprite shadow = lineShadows.get(i);
          shadow.setCommands(previousCommands);
        }
      }
    }

    // Only create a line if one doesn't exist.
    if (line == null) {
      line = new PathSprite();
      line.setStroke(stroke);
      chart.addSprite(line);

      if (chart.hasShadows()) {
        // create shadows
        for (int i = 0; i < shadowGroups.size(); i++) {
          PathSprite shadow = new PathSprite();
          Sprite shadowAttr = shadowAttributes.get(i);
          shadow.setStrokeWidth(shadowAttr.getStrokeWidth());
          shadow.setStrokeOpacity(shadowAttr.getStrokeOpacity());
          shadow.setStroke(shadowAttr.getStroke());
          shadow.setTranslation(new Translation(shadowAttr.getTranslation()));
          shadow.setFill(Color.NONE);
          shadow.setCommands(renderCommands);
          chart.addSprite(shadow);
          lineShadows.add(shadow);
        }
      }
    }
    if (stroke != null) {
      line.setStroke(stroke);
    }
    if (!Double.isNaN(strokeWidth)) {
      line.setStrokeWidth(strokeWidth);
    }
    line.setFill(Color.NONE);

    if (chart.isAnimated() && line.size() > 0) {
      if (markerIndex > 0) {
        if (smooth) {
          renderCommands.remove(1);
        } else {
          MoveTo move = (MoveTo) renderCommands.get(0);
          renderCommands.add(1, new LineTo(move.getX(), move.getY()));
        }
        previousCommands = renderCommands;
      }
      DrawFx.createCommandsAnimator(line, renderCommands).run(500);
    } else {
      line.setCommands(renderCommands);
      line.redraw();
    }

    if (lineRenderer != null) {
      lineRenderer.spriteRenderer(line, 0, chart.getStore());
    }

    if (fill != Color.NONE && fill != null) {
      fillCommands.addAll(renderCommands);
      fillCommands.add(new LineTo(x, bbox.getY() + bbox.getHeight()));
      fillCommands.add(new LineTo(bbox.getX(), bbox.getY() + bbox.getHeight()));
      fillCommands.add(new LineTo(bbox.getX(), firstY));

      if (fillSprite == null) {
        fillSprite = new PathSprite();
        fillSprite.setOpacity(0.3);
        fillSprite.setFill(fill);
        chart.addSprite(fillSprite);
      }
      if (chart.isAnimated() && fillSprite.size() > 0) {
        DrawFx.createCommandsAnimator(fillSprite, fillCommands).run(500);
      } else {
        fillSprite.setCommands(fillCommands);
        fillSprite.redraw();
      }
      if (fillRenderer != null) {
        fillRenderer.spriteRenderer(fillSprite, 0, chart.getStore());
      }
    }

    if (chart.hasShadows()) {
      for (int i = 0; i < lineShadows.size(); i++) {
        PathSprite shadow = lineShadows.get(i);
        if (chart.isAnimated()) {
          DrawFx.createCommandsAnimator(shadow, renderCommands).run(500);
        } else {
          shadow.setCommands(renderCommands);
          shadow.redraw();
        }
      }
    }
    drawLabels();
  }

  /**
   * Returns the fill color of the line.
   * 
   * @return the fill color of the line
   */
  public Color getFill() {
    return fill;
  }

  /**
   * Returns the {@link SeriesRenderer} used on the fill sprite.
   * 
   * @return the series renderer used on the fill sprite
   */
  public SeriesRenderer<M> getFillRenderer() {
    return fillRenderer;
  }

  /**
   * Returns the {@link SeriesRenderer} used on the line sprite.
   * 
   * @return the series renderer used on the line sprite
   */
  public SeriesRenderer<M> getLineRenderer() {
    return lineRenderer;
  }

  /**
   * Returns the marker index of the series. Determines the number of markers to
   * animate to simulate progression.
   * 
   * @return the marker index of the series
   */
  public double getMarkerIndex() {
    return markerIndex;
  }

  /**
   * Returns the number of segments of the smoothed line.
   * 
   * @return the number of segments of the smoothed line
   */
  public int getSegments() {
    return segments;
  }

  @Override
  public void hide(int yFieldIndex) {
    toggle(true);
  }

  @Override
  public void highlight(int yFieldIndex) {
    if (showMarkers) {
      highlighter.hightlight(sprites.get(yFieldIndex));
    }
  }

  @Override
  public void highlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.hightlight(sprites.get(i));
    }
  }

  /**
   * Returns whether or not markers are shown.
   * 
   * @return whether or not markers are shown
   */
  public boolean isShowMarkers() {
    return showMarkers;
  }

  /**
   * Returns whether or not the line is smoothed or straight.
   * 
   * @return true if smooth
   */
  public boolean isSmooth() {
    return smooth;
  }

  /**
   * Sets the fill color of the line. If none the fill will not be drawn.
   * 
   * @param fill the color of the fill
   */
  public void setFill(Color fill) {
    this.fill = fill;
  }

  /**
   * Sets the {@link SeriesRenderer} used on the fill sprite.
   * 
   * @param fillRenderer the series renderer used on the fill sprite
   */
  public void setFillRenderer(SeriesRenderer<M> fillRenderer) {
    this.fillRenderer = fillRenderer;
  }

  /**
   * Sets the {@link SeriesRenderer} used on the line sprite
   * 
   * @param lineRenderer the series renderer used on the line sprite
   */
  public void setLineRenderer(SeriesRenderer<M> lineRenderer) {
    this.lineRenderer = lineRenderer;
  }

  /**
   * Sets the marker index. Determines the number of markers to animate to
   * simulate progression.
   * 
   * @param markerIndex the marker index
   */
  public void setMarkerIndex(double markerIndex) {
    this.markerIndex = markerIndex;
  }

  /**
   * Sets the number of segments of the smoothed line. Smooth line enabled using
   * {@link #setSmooth(boolean)}.
   * 
   * @param segments the number of segments of the smoothed line
   */
  public void setSegments(int segments) {
    this.segments = segments;
  }

  /**
   * Sets whether or not to show markers.
   * 
   * @param showMarkers whether or not to show markers
   */
  public void setShowMarkers(boolean showMarkers) {
    if (this.showMarkers != showMarkers) {
      this.showMarkers = showMarkers;
      sprites.clear();
    }
  }

  /**
   * Sets whether or not the line is smoothed or straight.
   * 
   * @param smooth true if smooth
   */
  public void setSmooth(boolean smooth) {
    this.smooth = smooth;
  }

  @Override
  public void show(int yFieldIndex) {
    toggle(false);
  }

  @Override
  public void unHighlight(int yFieldIndex) {
    if (showMarkers) {
      highlighter.unHightlight(sprites.get(yFieldIndex));
    }
  }

  @Override
  public void unHighlightAll(int index) {
    for (int i = 0; i < sprites.size(); i++) {
      highlighter.unHightlight(sprites.get(i));
    }
  }

  @Override
  public boolean visibleInLegend(int index) {
    if (line == null) {
      return true;
    } else {
      return !line.isHidden();
    }
  }

  @Override
  protected int getIndex(PrecisePoint point) {
    if (line != null) {
      for (int i = 0; i < line.size(); i++) {
        PathCommand command = line.getCommand(i);
        PrecisePoint bound = getPointFromCommand(command);
        if (point.equalsNoPrecision(bound, selectionTolerance)) {
          return i;
        }
      }
    }
    return -1;
  }

  @Override
  protected Number getValue(int index) {
    return yField.getValue((chart.getCurrentStore().get(index)));
  }

  /**
   * Draws the labels on the series.
   */
  private void drawLabels() {
    if (labelConfig != null) {
      for (int i = 0; i < xValues.size(); i++) {
        final Sprite sprite;
        if (labels.get(i) != null) {
          sprite = labels.get(i);
        } else {
          sprite = labelConfig.getSpriteConfig().copy();
          labels.put(i, sprite);
          chart.addSprite(sprite);
        }
        setLabelText(sprite, i);
        PrecisePoint point = getPointFromCommand(lineCommands.get(i));
        double x = point.getX();
        double y = point.getY();
        if (showMarkers) {
          PreciseRectangle spriteBox = sprites.get(i).getBBox();
          PreciseRectangle textBox = sprite.getBBox();

          LabelPosition labelPosition = labelConfig.getLabelPosition();
          if (labelPosition == LabelPosition.START) {
            y -= textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
          } else if (labelPosition == LabelPosition.OUTSIDE) {
            y += textBox.getHeight() / 2.0 + spriteBox.getHeight() / 2.0;
          }
        }
        if (chart.isAnimated() && sprite.getTranslation() != null) {
          DrawFx.createTranslationAnimator(sprite, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
        } else {
          sprite.setTranslation(x, y);
          sprite.redraw();
        }
      }
    }
  }

  /**
   * Toggles all the sprites in the series to be hidden or shown.
   * 
   * @param hide if true hides
   */
  private void toggle(boolean hide) {
    if (line != null) {
      line.setHidden(hide);
      line.redraw();
      for (int i = 0; i < lineShadows.size(); i++) {
        lineShadows.get(i).setHidden(hide);
        lineShadows.get(i).redraw();
      }
    }
    if (fillSprite != null) {
      fillSprite.setHidden(hide);
      fillSprite.redraw();
    }
    if (sprites.size() > 0) {
      for (int i = 0; i < sprites.size(); i++) {
        sprites.get(i).setHidden(hide);
        sprites.get(i).redraw();
      }
    }
  }

}
