/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sencha.gxt.chart.client.chart.Legend;
import com.sencha.gxt.chart.client.chart.axis.Axis;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.DrawFx;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.Translation;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.RectangleSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.data.shared.ListStore;

/**
 * Creates a Bar Chart. A Bar Chart is a useful visualization technique to
 * display quantitative information for different categories that can show some
 * progression (or regression) in the data set.
 * 
 * @param <M> the data type used by this series
 */
public class BarSeries<M> extends MultipleColorSeries<M> {

  private boolean column = false;
  private boolean stacked = false;

  // The gutter space between single bars, as a percentage of the bar width
  private double gutter = 38.2;

  // The gutter space between groups of bars, as a percentage of the bar width
  private double groupGutter = 38.2;

  // Padding between the left/right axes and the bars
  protected int xPadding = 0;

  // Padding between the top/bottom axes and the bars
  protected int yPadding = 10;

  private Axis<M, ?> axis;
  private double minY = 0;
  private double maxY = 0;
  private double groupBarWidth = 0;
  private int groupBarsLength = 0;
  private double scale = 0;
  private double zero = 0;
  private Set<Integer> exclude = new HashSet<Integer>();
  // the bar attributes
  private ArrayList<RectangleSprite> rects = new ArrayList<RectangleSprite>();
  private Map<Integer, Double> totalPositiveDimensions = new HashMap<Integer, Double>();
  private Map<Integer, Double> totalNegativeDimensions = new HashMap<Integer, Double>();
  protected List<ValueProvider<M, ? extends Number>> yFields = new ArrayList<ValueProvider<M, ? extends Number>>();

  /**
   * Creates a bar {@link Series}.
   */
  public BarSeries() {
    // setup shadow attributes
    Sprite config = new PathSprite();
    config.setStrokeWidth(6);
    config.setStrokeOpacity(0.05);
    config.setStroke(new RGB(200, 200, 200));
    config.setTranslation(1.2, 1.2);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(4);
    config.setStrokeOpacity(0.1);
    config.setStroke(new RGB(150, 150, 150));
    config.setTranslation(0.9, 0.9);
    shadowAttributes.add(config);
    config = new PathSprite();
    config.setStrokeWidth(2);
    config.setStrokeOpacity(0.15);
    config.setStroke(new RGB(100, 100, 100));
    config.setTranslation(0.6, 0.6);
    shadowAttributes.add(config);

    setHighlighter(new BarHighlighter());
  }

  /**
   * Adds a data field for the y axis of the series.
   * 
   * @param yField the value provider for the data on the y axis
   */
  public void addYField(ValueProvider<M, ? extends Number> yField) {
    this.yFields.add(yField);
  }

  @Override
  public void drawSeries() {
    ListStore<M> store = chart.getCurrentStore();

    if (store == null || store.size() == 0) {
      return;
    }

    calculatePaths();

    // Create new or reuse sprites and animate/display
    for (int i = 0; i < rects.size(); i++) {
      if (chart.hasShadows()) {
        renderShadows(i);
      }

      final RectangleSprite bar;
      if (i < sprites.size()) {
        bar = (RectangleSprite) sprites.get(i);
      } else {
        // Create a new bar if needed (no height)
        bar = new RectangleSprite();
        sprites.add(bar);
        chart.addSprite(bar);
      }
      RectangleSprite rect = rects.get(i);
      bar.setFill(rect.getFill());
      if (stroke != null) {
        bar.setStroke(stroke);
      }
      if (!Double.isNaN(strokeWidth)) {
        bar.setStrokeWidth(strokeWidth);
      }
      if (chart.isAnimated() && !Double.isNaN(bar.getX())) {
        DrawFx.createRectangleAnimator(bar, rect.toRectangle()).run(chart.getAnimationDuration(),
            chart.getAnimationEasing());
      } else {
        bar.setX(rect.getX());
        bar.setY(rect.getY());
        bar.setWidth(rect.getWidth());
        bar.setHeight(rect.getHeight());
        bar.redraw();
      }
      if (renderer != null) {
        renderer.spriteRenderer(bar, i, store);
      }
    }
    drawLabels();
  }

  /**
   * Calculates the girth of bars in the series.
   * 
   * @return the girth of bars in the series
   */
  public double getBarGirth() {
    int ln = chart.getCurrentStore().size();
    double gutter = this.gutter / 100;
    double chartGirth = column ? chart.getBBox().getWidth() : chart.getBBox().getHeight();
    double numerator = (column ? xPadding : yPadding) * 2;
    double denominator = (ln * (gutter + 1) - gutter);
    return (chartGirth - numerator) / denominator;
  }

  /**
   * Returns the gutter between group bars.
   * 
   * @return the gutter between group bars
   */
  public double getGroupGutter() {
    return groupGutter;
  }

  /**
   * Returns the gutter between bars.
   * 
   * @return the gutter between bars
   */
  public double getGutter() {
    return gutter;
  }

  @Override
  public double[] getGutters() {
    double gutter = Math.ceil((column ? xPadding : yPadding) + getBarGirth() / 2);
    double[] gutters = {0, 0};
    gutters[column ? 0 : 1] = gutter;
    return gutters;
  }

  @Override
  public ArrayList<String> getLegendTitles() {
    ArrayList<String> titles = new ArrayList<String>();
    for (int j = 0; j < getyFields().size(); j++) {
      if (legendTitles.size() > j) {
        titles.add(legendTitles.get(j));
      } else {
        titles.add(getValueProviderName(getyField(j)));
      }
    }
    return titles;
  }

  /**
   * Returns the value provider for the y-axis of the series at the given index.
   * 
   * @param index the index of the value provider
   * @return the value provider for the y-axis of the series at the given index
   */
  public ValueProvider<M, ? extends Number> getyField(int index) {
    return yFields.get(index);
  }

  /**
   * Returns the list of value providers for the y-axis of the series.
   * 
   * @return the list of value providers for the y-axis of the series
   */
  public List<ValueProvider<M, ? extends Number>> getyFields() {
    return yFields;
  }

  @Override
  public void hide(int yFieldIndex) {
    if (yFields.size() > 1) {
      for (int i = 0; i < sprites.size() / yFields.size(); i++) {
        sprites.get(yFieldIndex + i * yFields.size()).setHidden(true);
        sprites.get(yFieldIndex + i * yFields.size()).redraw();
      }
      exclude.add(yFieldIndex);
      drawSeries();
    } else {
      toggle(true);
    }
    // update axes
    chart.getAxis(yAxisPosition);
  }

  @Override
  public void highlight(int yFieldIndex) {
    RectangleSprite bar = (RectangleSprite) sprites.get(yFieldIndex);
    highlighter.hightlight(bar);
  }

  @Override
  public void highlightAll(int index) {
    if (yFields.size() > 1) {
      for (int i = 0; i < sprites.size() / yFields.size(); i++) {
        highlighter.hightlight(sprites.get(index + i * yFields.size()));
      }
    } else {
      for (int i = 0; i < sprites.size(); i++) {
        highlighter.hightlight(sprites.get(i));
      }
    }
  }

  /**
   * Returns whether or not the series is a column series.
   * 
   * @return true if the series is a column series
   */
  public boolean isColumn() {
    return column;
  }

  /**
   * Returns whether or not the series is stacked.
   * 
   * @return whether or not the series is stacked
   */
  public boolean isStacked() {
    return stacked;
  }

  /**
   * Sets whether or not the series is a column series.
   * 
   * @param column true if the series is a column series
   */
  public void setColumn(boolean column) {
    this.column = column;
    if (column) {
      xPadding = 10;
      yPadding = 0;
    } else {
      xPadding = 0;
      yPadding = 10;
    }
  }

  /**
   * Sets the gutter between group bars.
   * 
   * @param groupGutter
   */
  public void setGroupGutter(double groupGutter) {
    this.groupGutter = groupGutter;
  }

  /**
   * Sets the gutter between bars.
   * 
   * @param gutter the gutter between bars
   */
  public void setGutter(double gutter) {
    this.gutter = gutter;
  }

  /**
   * Sets the series title used in the legend.
   * 
   * @param title the series title used in the legend
   */
  public void setLegendTitle(String title) {
    legendTitles.clear();
    legendTitles.add(title);

    if (chart != null) {
      Legend<M> legend = chart.getLegend();
      if (legend != null) {
        legend.create();
        legend.updatePosition();
      }
    }
  }

  /**
   * Sets the list of labels used by the legend.
   * 
   * @param legendTitles the list of labels
   */
  public void setLegendTitles(List<String> legendTitles) {
    this.legendTitles = legendTitles;

    if (chart != null) {
      Legend<M> legend = chart.getLegend();
      if (legend != null) {
        legend.create();
        legend.updatePosition();
      }
    }
  }

  /**
   * Sets whether or not the series is stacked.
   * 
   * @param stacked whether or not the series is stacked
   */
  public void setStacked(boolean stacked) {
    this.stacked = stacked;
  }

  @Override
  public void show(int yFieldIndex) {
    if (yFields.size() > 1) {
      for (int i = 0; i < sprites.size() / yFields.size(); i++) {
        sprites.get(yFieldIndex + i * yFields.size()).setHidden(false);
        sprites.get(yFieldIndex + i * yFields.size()).redraw();
      }
      exclude.remove(yFieldIndex);
      calculateBBox(false);
      drawSeries();
    } else {
      toggle(false);
    }
  }

  @Override
  public void unHighlight(int yFieldIndex) {
    RectangleSprite bar = (RectangleSprite) sprites.get(yFieldIndex);
    highlighter.unHightlight(bar);
  }

  @Override
  public void unHighlightAll(int index) {
    if (yFields.size() > 1) {
      for (int i = 0; i < sprites.size() / yFields.size(); i++) {
        highlighter.unHightlight(sprites.get(index + i * yFields.size()));
      }
    } else {
      for (int i = 0; i < sprites.size(); i++) {
        highlighter.unHightlight(sprites.get(i));
      }
    }
  }

  @Override
  public boolean visibleInLegend(int index) {
    if (yFields.size() > 1) {
      if (exclude.contains(index)) {
        return false;
      }
      return true;
    } else {
      if (sprites.size() == 0) {
        return true;
      } else {
        return !sprites.get(0).isHidden();
      }
    }
  }

  @Override
  protected int getIndex(PrecisePoint point) {
    for (int i = 0; i < sprites.size(); i++) {
      if (((RectangleSprite) sprites.get(i)).toRectangle().contains(point)) {
        return i;
      }
    }
    return -1;
  }

  @Override
  protected Number getValue(int index) {
    Number value = null;
    if (yFields.size() > 1) {
      int storeIndex = (int) Math.floor(index / yFields.size());
      int yFieldIndex = index - (yFields.size() * storeIndex);
      value = yFields.get(yFieldIndex).getValue(chart.getCurrentStore().get(storeIndex));

    } else if (yFields.size() == 1) {
      value = yFields.get(0).getValue(chart.getCurrentStore().get(index));
    }
    return value;
  }

  /**
   * Calculates the bounds of the series.
   */
  private void calculateBounds() {
    ListStore<M> store = chart.getCurrentStore();
    double barWidth = getBarGirth();
    double groupGutter = this.groupGutter / 100;
    groupBarsLength = yFields.size();
    PreciseRectangle chartBBox = chart.getBBox();

    calculateBBox(false);

    // skip excluded series
    for (int i = 0; i < exclude.size(); i++) {
      if (exclude.contains(i)) {
        groupBarsLength--;
      }
    }

    if (yAxisPosition != null) {
      axis = chart.getAxis(yAxisPosition);
      if (axis != null) {
        minY = axis.getFrom();
        maxY = axis.getTo();
      }
    }

    // TODO: axis error

    scale = (column ? chartBBox.getHeight() - yPadding * 2 : chartBBox.getWidth() - xPadding * 2) / (maxY - minY);
    groupBarWidth = barWidth / ((stacked ? 1 : groupBarsLength) * (groupGutter + 1) - groupGutter);
    zero = column ? chartBBox.getY() + chartBBox.getHeight() - yPadding : chartBBox.getX() + xPadding;

    List<Double> totalPositive = new ArrayList<Double>();
    List<Double> totalNegative = new ArrayList<Double>();
    if (stacked) {
      for (int i = 0; i < store.size(); i++) {
        M model = store.get(i);
        totalPositive.add(0.0);
        totalNegative.add(0.0);
        for (int j = 0; j < yFields.size(); j++) {
          double value = yFields.get(j).getValue(model).doubleValue();
          if (value > 0) {
            totalPositive.set(i, totalPositive.get(i) + value);
          } else {
            totalNegative.set(i, totalNegative.get(i) + Math.abs(value));
          }
        }
      }
      if (maxY > 0) {
        totalPositive.add(maxY);
      } else {
        totalNegative.add(Math.abs(maxY));
      }
      if (minY > 0) {
        totalPositive.add(minY);
      } else {
        totalNegative.add(Math.abs(minY));
      }
      double minus = 0;
      double plus = 0;
      for (int i = 0; i < totalNegative.size(); i++) {
        minus = Math.max(minus, totalNegative.get(i));
      }
      for (int i = 0; i < totalPositive.size(); i++) {
        plus = Math.max(plus, totalPositive.get(i));
      }
      scale = (column ? bbox.getHeight() - yPadding * 2 : bbox.getWidth() - xPadding * 2) / (plus + minus);
      zero = zero + minus * scale * (column ? -1 : 1);
    } else if (minY / maxY < 0) {
      zero = zero - minY * scale * (column ? -1 : 1);
    }
  }

  /**
   * Build an array of paths for the chart.
   */
  private void calculatePaths() {
    ListStore<M> store = chart.getCurrentStore();
    double totalDimension = 0;
    double totalNegativeDimension = 0;
    double gutter = this.gutter / 100;
    double groupGutter = this.groupGutter / 100;
    int fieldsLength = yFields.size();

    calculateBounds();

    for (int i = 0; i < store.size(); i++) {
      int counter = 0;
      double bottom = zero;
      double top = zero;
      for (int j = 0; j < fieldsLength; j++) {
        // Excluded series
        if (exclude.contains(j)) {
          continue;
        }
        double value = yFields.get(j).getValue(store.get(i)).doubleValue();
        double height = Math.round((value - ((minY < 0) ? 0 : minY)) * scale);
        RectangleSprite rect;
        if ((i * fieldsLength) + j < rects.size()) {
          rect = rects.get((i * fieldsLength) + j);
        } else {
          // Create a new sprite if needed (no height)
          rect = new RectangleSprite();
          rects.add(rect);
        }
        if (colors.size() > 0) {
          rect.setFill(colors.get((yFields.size() > 1 ? j : 0) % colors.size()));
        }
        if (column) {
          rect.setHeight(height);
          rect.setWidth(Math.max(groupBarWidth, 0));
          rect.setX(bbox.getX() + xPadding + i * getBarGirth() * (1 + gutter) + counter * groupBarWidth
              * (1 + groupGutter) * (!stacked ? 1 : 0));
          rect.setY(bottom - height);
        } else {
          // draw in reverse order
          rect.setHeight(Math.max(groupBarWidth, 0));
          rect.setWidth(height + (bottom == zero ? 1 : 0));
          rect.setX(bottom + (bottom != zero ? 1 : 0));
          rect.setY(bbox.getY() + yPadding + (store.size() - 1 - i) * getBarGirth() * (1 + gutter) + counter
              * groupBarWidth * (1 + groupGutter) * (!stacked ? 1 : 0) + 1);
        }
        if (height < 0) {
          if (column) {
            rect.setY(top);
            rect.setHeight(Math.abs(height));
          } else {
            rect.setX(top + height);
            rect.setWidth(Math.abs(height));
          }
        }
        if (stacked) {
          if (height < 0) {
            top += height * (column ? -1 : 1);
          } else {
            bottom += height * (column ? -1 : 1);
          }
          totalDimension += Math.abs(height);
          if (height < 0) {
            totalNegativeDimension += Math.abs(height);
          }
        }
        rect.setX(Math.floor(rect.getX()) + 1);
        rect.setY(Math.floor(rect.getY()));
        // TODO: IE9 specific fix
        rect.setWidth(Math.floor(rect.getWidth()));
        rect.setHeight(Math.floor(rect.getHeight()));
        counter++;
      }
      if (stacked) {
        totalPositiveDimensions.put(i * counter, totalDimension);
        totalNegativeDimensions.put(i * counter, totalNegativeDimension);
      }
    }
  }

  /**
   * Draws the labels on the series.
   */
  private void drawLabels() {
    if (labelConfig != null) {
      LabelPosition labelPosition = labelConfig.getLabelPosition();
      for (int j = 0; j < rects.size(); j++) {
        if (exclude.contains(j)) {
          continue;
        }
        final Sprite sprite;
        if (labels.get(j) != null) {
          sprite = labels.get(j);
        } else {
          sprite = labelConfig.getSpriteConfig().copy();
          labels.put(j, sprite);
          chart.addSprite(sprite);
        }
        setLabelText(sprite, j);
        PreciseRectangle box = sprite.getBBox();
        RectangleSprite bar = rects.get(j);
        double x = 0;
        double y = 0;
        if (column) {
          x = bar.getX() + bar.getWidth() / 2.0;
          if (labelPosition == LabelPosition.START) {
            y = bar.getY();
          } else if (labelPosition == LabelPosition.END) {
            if (bar.getHeight() > box.getHeight()) {
              y = bar.getY() + box.getHeight() / 2.0;
            } else {
              y = bar.getY() - box.getHeight() / 2.0;
            }
          } else if (labelPosition == LabelPosition.OUTSIDE) {
            y = bar.getY() + bar.getHeight();
          }
        } else {
          y = bar.getY() + bar.getHeight() - box.getHeight() / 2.0;
          if (labelPosition == LabelPosition.START) {
            x = bar.getX();
          } else if (labelPosition == LabelPosition.END) {
            if (bar.getWidth() > box.getWidth()) {
              x = bar.getX() + bar.getWidth() - box.getWidth();
            } else {
              x = bar.getX() + bar.getWidth();
            }
          } else if (labelPosition == LabelPosition.OUTSIDE) {
            x = bar.getX() + bar.getWidth();
          }
        }
        if (chart.isAnimated() && sprite.getTranslation() != null) {
          DrawFx.createTranslationAnimator(sprite, x, y).run(chart.getAnimationDuration(), chart.getAnimationEasing());
        } else {
          sprite.setTranslation(x, y);
        }
        SeriesRenderer<M> labelRenderer = labelConfig.getSpriteRenderer();
        if (labelRenderer != null) {
          labelRenderer.spriteRenderer(sprite, j, chart.getCurrentStore());
        }
        sprite.redraw();
      }
    }
  }

  /**
   * Renders the shadows for the bar at the given index.
   * 
   * @param index the index of the shadows
   */
  private void renderShadows(int i) {
    if (groupBarsLength == 0) {
      return;
    }
    if ((stacked && (i % groupBarsLength) == 0) || !stacked) {
      int j = i / groupBarsLength;
      // create shadows
      for (int shindex = 0; shindex < shadowGroups.size(); shindex++) {
        Sprite shadowBarAttr = shadowAttributes.get(shindex);
        int index = stacked ? j : i;
        SpriteList<Sprite> shadows = shadowGroups.get(shindex);
        final RectangleSprite shadowSprite;
        if (index < shadows.size()) {
          shadowSprite = (RectangleSprite) shadows.get(index);
        } else {
          shadowSprite = new RectangleSprite();
          shadowSprite.setStrokeWidth(shadowBarAttr.getStrokeWidth());
          shadowSprite.setStrokeOpacity(shadowBarAttr.getStrokeOpacity());
          shadowSprite.setStroke(shadowBarAttr.getStroke());
          shadowSprite.setFill(Color.NONE);
          shadowSprite.setTranslation(new Translation(shadowBarAttr.getTranslation()));
          shadows.add(shadowSprite);
          chart.addSprite(shadowSprite);
        }
        RectangleSprite rect = rects.get(shindex);
        if (stacked) {
          double totalPositiveDimension = totalPositiveDimensions.get(i);
          double totalNegativeDimension = totalNegativeDimensions.get(i);
          if (column) {
            rect.setY(zero - totalNegativeDimension);
            rect.setHeight(totalPositiveDimension);
          } else {
            rect.setX(zero - totalNegativeDimension);
            rect.setWidth(totalPositiveDimension);
          }
        }
        if (chart.isAnimated() && !Double.isNaN(shadowSprite.getHeight()) && !Double.isNaN(shadowSprite.getWidth())) {
          DrawFx.createRectangleAnimator(shadowSprite, rect.toRectangle()).run(chart.getAnimationDuration(),
              chart.getAnimationEasing());
        } else {
          shadowSprite.setX(rect.getX());
          shadowSprite.setY(rect.getY());
          shadowSprite.setWidth(rect.getWidth());
          shadowSprite.setHeight(rect.getHeight());
          shadowSprite.redraw();
        }
      }
    }
  }

  /**
   * Toggles all the sprites in the series to be hidden or shown.
   * 
   * @param hide if true hides
   */
  private void toggle(boolean hide) {
    calculateBBox(false);
    if (sprites.size() > 0) {
      for (int i = 0; i < sprites.size(); i++) {
        sprites.get(i).setHidden(hide);
        sprites.get(i).redraw();
      }
    }
    for (int i = 0; i < shadowGroups.size(); i++) {
      SpriteList<Sprite> shadows = shadowGroups.get(i);
      for (int j = 0; j < shadows.size(); j++) {
        shadows.get(j).setHidden(true);
        shadows.get(j).redraw();
      }
    }
  }
}
