/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart.series;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.axis.Axis;
import com.sencha.gxt.chart.client.draw.DrawFx;
import com.sencha.gxt.chart.client.draw.path.LineTo;
import com.sencha.gxt.chart.client.draw.path.MoveTo;
import com.sencha.gxt.chart.client.draw.path.PathCommand;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.SpriteList;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.data.shared.LabelProvider;
import com.sencha.gxt.data.shared.ListStore;

/**
 * Creates a Stacked Area Chart. The stacked area chart is useful when
 * displaying multiple aggregated layers of information.
 * 
 * @param <M> the data type used by this series
 */
public class AreaSeries<M> extends MultipleColorSeries<M> {

  private double xScale;
  private double yScale;
  private PrecisePoint min;
  private PrecisePoint max;
  private SpriteList<PathSprite> areas = new SpriteList<PathSprite>();
  private Set<Integer> exclude = new HashSet<Integer>();
  private List<Double> xValues = new ArrayList<Double>();
  private List<double[]> yValues = new ArrayList<double[]>();
  private Map<Integer, ArrayList<PathCommand>> areasCommands = new HashMap<Integer, ArrayList<PathCommand>>();
  private List<ValueProvider<M, ? extends Number>> yFields = new ArrayList<ValueProvider<M, ? extends Number>>();

  /**
   * Creates an area {@link Series}.
   */
  public AreaSeries() {
    setHighlighter(new AreaHighlighter());
  }

  /**
   * Adds a data field for the y axis of the series.
   * 
   * @param yField the value provider for the data on the y axis
   */
  public void addYField(ValueProvider<M, ? extends Number> yField) {
    this.yFields.add(yField);
  }

  @Override
  public void drawSeries() {
    calculateBounds();
    calculatePaths();

    for (int i = 0; i < yFields.size(); i++) {
      if (exclude.contains(i)) {
        continue;
      }
      final PathSprite path;
      if (i < areas.size()) {
        path = areas.get(i);
      } else {
        path = new PathSprite();
        path.setFill(colors.get(i));
        chart.addSprite(path);
        areas.add(path);
      }
      if (chart.isAnimated() && path.getCommands().size() > 0) {
        DrawFx.createCommandsAnimator(path, areasCommands.get(i)).run(chart.getAnimationDuration(),
            chart.getAnimationEasing());
      } else {
        path.setCommands(areasCommands.get(i));
        path.redraw();
      }
      if (stroke != null) {
        path.setStroke(stroke);
      }
      if (!Double.isNaN(strokeWidth)) {
        path.setStrokeWidth(strokeWidth);
      }
      if (renderer != null) {
        renderer.spriteRenderer(path, i, chart.getCurrentStore());
      }
    }

    drawLabels();
  }

  @Override
  public ArrayList<String> getLegendTitles() {
    ArrayList<String> titles = new ArrayList<String>();
    for (int j = 0; j < getYFields().size(); j++) {
      if (legendTitles.size() > j) {
        titles.add(legendTitles.get(j));
      } else {
        titles.add(getValueProviderName(getYFields().get(j)));
      }
    }
    return titles;
  }

  /**
   * Returns the value provider for the y-axis of the series at the given index.
   * 
   * @param index the index of the value provider
   * @return the value provider for the y-axis of the series at the given index
   */
  public ValueProvider<M, ? extends Number> getYField(int index) {
    return yFields.get(index);
  }

  /**
   * Returns the list of value providers for the y-axis of the series.
   * 
   * @return the list of value providers for the y-axis of the series
   */
  public List<ValueProvider<M, ? extends Number>> getYFields() {
    return yFields;
  }

  @Override
  public void hide(int yFieldIndex) {
    areas.get(yFieldIndex).setHidden(true);
    areas.get(yFieldIndex).redraw();
    exclude.add(yFieldIndex);
    drawSeries();
  }

  @Override
  public void highlight(int yFieldIndex) {
    if (areas.size() > yFieldIndex) {
      highlighter.hightlight(areas.get(yFieldIndex));
    }
  }

  @Override
  public void highlightAll(int index) {
    highlight(index);
  }

  /**
   * Sets the list of labels used by the legend.
   * 
   * @param legendTitles the list of labels
   */
  public void setLegendTitles(List<String> legendTitles) {
    this.legendTitles = legendTitles;
  }

  @Override
  public void show(int yFieldIndex) {
    areas.get(yFieldIndex).setHidden(false);
    areas.get(yFieldIndex).redraw();
    exclude.remove(yFieldIndex);
    drawSeries();
  }

  @Override
  public void unHighlight(int yFieldIndex) {
    if (areas.size() > yFieldIndex) {
      highlighter.unHightlight(areas.get(yFieldIndex));
    }
  }

  @Override
  public void unHighlightAll(int index) {
    unHighlight(index);
  }

  @Override
  public boolean visibleInLegend(int index) {
    if (exclude.contains(index)) {
      return false;
    }
    return true;
  }

  @Override
  protected int getIndex(PrecisePoint point) {
    for (int i = 0; i < areas.size(); i++) {
      if (exclude.contains(i)) {
        continue;
      }
      PathSprite area = areas.get(i);
      List<PathCommand> commands = area.getCommands();
      double dist = Double.POSITIVE_INFINITY;
      for (int j = 1; j < chart.getCurrentStore().size(); j++) {
        PrecisePoint bound = getPointFromCommand(area.getCommand(j));
        if (Double.compare(dist, Math.abs(point.getX() - bound.getX())) > 0) {
          dist = Math.abs(point.getX() - bound.getX());
        } else {
          bound = getPointFromCommand(area.getCommand(j - 1));
          if (point.getY() >= bound.getY()
              && (i == 0 || point.getY() <= getPointFromCommand(commands.get(commands.size() - j - 1)).getY())) {
            return i;
          }
          break;
        }
      }
    }
    return -1;
  }

  @Override
  protected Number getValue(int index) {
    ListStore<M> store = chart.getCurrentStore();
    double total = 0;
    for (int i = 0; i < store.size(); i++) {
      if (exclude.contains(i)) {
        continue;
      }
      total += yFields.get(index).getValue(store.get(i)).doubleValue();
    }
    return total;
  }

  /**
   * Calculates the bounds of the series.
   */
  private void calculateBounds() {
    PreciseRectangle chartBBox = chart.getBBox();
    ListStore<M> store = chart.getCurrentStore();
    min = new PrecisePoint(Double.NaN, Double.NaN);
    max = new PrecisePoint(Double.NaN, Double.NaN);
    // get bounding box dimensions
    bbox.setX(chartBBox.getX());
    bbox.setY(chartBBox.getY());
    bbox.setWidth(chartBBox.getWidth());
    bbox.setHeight(chartBBox.getHeight());
    xValues = new ArrayList<Double>();
    yValues = new ArrayList<double[]>();

    Axis<M, ?> axis = chart.getAxis(yAxisPosition);
    if (axis != null) {
      if (axis.getPosition() == Position.TOP || axis.getPosition() == Position.BOTTOM) {
        min.setX(axis.getFrom());
        max.setX(axis.getTo());
      } else {
        min.setY(axis.getFrom());
        max.setY(axis.getTo());
      }
    }

    if (Double.isNaN(min.getX())) {
      min.setX(0);
      xScale = bbox.getWidth() / (chart.getCurrentStore().size() - 1);
    } else {
      xScale = bbox.getWidth() / (max.getX() - min.getX());
    }

    if (Double.isNaN(min.getY())) {
      min.setY(0);
      yScale = bbox.getHeight() / (chart.getCurrentStore().size() - 1);
    } else {
      yScale = bbox.getHeight() / (max.getY() - min.getY());
    }

    for (int i = 0; i < store.size(); i++) {
      M model = store.get(i);
      double xValue = 0;
      double yElement;
      double[] yValue = new double[yFields.size()];
      double acumY = 0;
      if (i == 0) {
        max.setX(0);
        max.setY(0);
      }

      // Ensure a value
      if (xField == null) {
        xValue = i;
      } else {
        xValue = xField.getValue(model).doubleValue();
      }
      xValues.add(xValue);

      for (int j = 0; j < yFields.size(); j++) {
        yElement = yFields.get(j).getValue(model).doubleValue();
        acumY += yElement;
        yValue[j] = yElement;
      }
      max.setX(Math.max(max.getX(), xValue));
      max.setY(Math.max(max.getY(), acumY));
      yValues.add(yValue);
    }
    xScale = bbox.getWidth() / (max.getX() - min.getX());
    yScale = bbox.getHeight() / (max.getY() - min.getY());
  }

  /**
   * Build a list of paths for the chart.
   */
  private void calculatePaths() {
    boolean first = true;
    double xValue;
    double[] yValue;
    double x = 0;
    double y = 0;
    double acumY;
    areasCommands = new HashMap<Integer, ArrayList<PathCommand>>();
    Map<Integer, ArrayList<PathCommand>> areasComponentCommands = new HashMap<Integer, ArrayList<PathCommand>>();
    ArrayList<PathCommand> commands = null;
    ArrayList<PathCommand> componentCommands = null;
    PathCommand command;

    int ln = chart.getCurrentStore().size();

    // Start the path
    for (int i = 0; i < ln; i++) {
      componentCommands = new ArrayList<PathCommand>();
      xValue = xValues.get(i);
      yValue = yValues.get(i);
      x = bbox.getX() + (xValue - min.getX()) * xScale;
      acumY = 0;

      for (int j = 0; j < yValue.length; j++) {

        if (exclude.contains(j)) {
          continue;
        }

        if (areasComponentCommands.get(j) == null) {
          componentCommands = new ArrayList<PathCommand>();
          areasComponentCommands.put(j, componentCommands);
        } else {
          componentCommands = areasComponentCommands.get(j);
        }
        acumY += yValue[j];
        y = bbox.getY() + bbox.getHeight() - (acumY - min.getY()) * yScale;
        if (areasCommands.get(j) == null) {
          commands = new ArrayList<PathCommand>();
          commands.add(new MoveTo(x, y));
          areasCommands.put(j, commands);
        } else {
          commands = areasCommands.get(j);
          commands.add(new LineTo(x, y));
        }
        componentCommands.add(new LineTo(x, y));
      }
    }

    int prevAreaIndex = 0;
    // Close the paths
    for (int i = 0; i < yFields.size(); i++) {

      if (exclude.contains(i)) {
        continue;
      }

      commands = areasCommands.get(i);
      // Close bottom path to the axis
      if (first) {
        first = false;
        commands.add(new LineTo(x, bbox.getY() + bbox.getHeight()));
        commands.add(new LineTo(bbox.getX(), bbox.getY() + bbox.getHeight()));
      }
      // Close other paths to the one before them
      else {
        componentCommands = areasComponentCommands.get(prevAreaIndex);
        // reverse the componentCommands
        for (int j = 0; j < componentCommands.size() / 2; j++) {
          command = componentCommands.remove(j);
          componentCommands.add(componentCommands.size() - j, command);
          command = componentCommands.remove(componentCommands.size() - j - 2);
          componentCommands.add(j, command);
        }
        command = componentCommands.get(0);
        if (command instanceof MoveTo) {
          commands.add(new LineTo(x, ((MoveTo) command).getY()));
        } else if (command instanceof LineTo) {
          commands.add(new LineTo(x, ((LineTo) command).getY()));
        }

        for (int j = 0; j < ln; j++) {
          command = componentCommands.get(j);
          if (command instanceof MoveTo) {
            commands.add(new MoveTo((MoveTo) command));
          } else if (command instanceof LineTo) {
            commands.add(new LineTo((LineTo) command));
          }
        }
        command = commands.get(0);
        if (command instanceof MoveTo) {
          commands.add(new LineTo(bbox.getX(), ((MoveTo) command).getY()));
        } else if (command instanceof LineTo) {
          commands.add(new LineTo(bbox.getX(), ((LineTo) command).getY()));
        }
      }
      prevAreaIndex = i;
    }
  }

  /**
   * Draw the labels on the series.
   */
  private void drawLabels() {
    if (labelConfig != null) {
      LabelPosition labelPosition = labelConfig.getLabelPosition();
      if (labelPosition == LabelPosition.OUTSIDE) {
        int top = areasCommands.size() - 1;
        for (int i = 0; i < chart.getCurrentStore().size(); i++) {
          final Sprite sprite;
          if (labels.get(i) != null) {
            sprite = labels.get(i);
          } else {
            sprite = labelConfig.getSpriteConfig().copy();
            labels.put(i, sprite);
            chart.addSprite(sprite);
          }
          setLabelText(sprite, i);
          PrecisePoint point = getPointFromCommand(areasCommands.get(top).get(i));
          if (chart.isAnimated() && sprite.getTranslation() != null) {
            DrawFx.createTranslationAnimator(sprite, point.getX(), point.getY()).run(chart.getAnimationDuration(),
                chart.getAnimationEasing());
          } else {
            sprite.setTranslation(point.getX(), point.getY());
            sprite.redraw();
          }
        }
      } else if (labelPosition == LabelPosition.END || labelPosition == LabelPosition.START) {
        for (int i = 0; i < yValues.size(); i++) {
          double[] values = yValues.get(i);
          for (int j = 0; j < values.length; j++) {
            final Sprite sprite;
            if (labels.get(i * values.length + j) != null) {
              sprite = labels.get(i * values.length + j);
            } else {
              sprite = labelConfig.getSpriteConfig().copy();
              labels.put(i * values.length + j, sprite);
              chart.addSprite(sprite);
            }
            if (sprite instanceof TextSprite) {
              TextSprite text = (TextSprite) sprite;
              LabelProvider<Number> numeric = labelConfig.getNumericLabelProvider();
              LabelProvider<M> custom = labelConfig.getCustomLabelProvider();
              if (numeric != null) {
                text.setText(numeric.getLabel(values[j]));
              } else if (custom != null) {
                text.setText(custom.getLabel(chart.getCurrentStore().get(j)));
              }
              text.redraw();
            }
            double offsetY = 0;
            if (labelPosition == LabelPosition.START) {
              offsetY = sprite.getBBox().getHeight();
            }
            PrecisePoint point = getPointFromCommand(areasCommands.get(j).get(i));
            if (chart.isAnimated() && sprite.getTranslation() != null) {
              DrawFx.createTranslationAnimator(sprite, point.getX(), point.getY() + offsetY).run(
                  chart.getAnimationDuration(), chart.getAnimationEasing());
            } else {
              sprite.setTranslation(point.getX(), point.getY() + offsetY);
              sprite.redraw();
            }
          }
        }
      }
    }
  }
}
