/**
 * Ext GWT 3.0.0-beta1 - Ext for GWT
 * Copyright(c) 2007-2011, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.chart;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.event.shared.HandlerRegistration;
import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.event.LegendHandler;
import com.sencha.gxt.chart.client.chart.event.LegendHandler.HasLegendHandlers;
import com.sencha.gxt.chart.client.chart.event.LegendItemOutEvent;
import com.sencha.gxt.chart.client.chart.event.LegendItemOutEvent.HasLegendItemOutHandlers;
import com.sencha.gxt.chart.client.chart.event.LegendItemOutEvent.LegendItemOutHandler;
import com.sencha.gxt.chart.client.chart.event.LegendItemOverEvent;
import com.sencha.gxt.chart.client.chart.event.LegendItemOverEvent.HasLegendItemOverHandlers;
import com.sencha.gxt.chart.client.chart.event.LegendItemOverEvent.LegendItemOverHandler;
import com.sencha.gxt.chart.client.chart.event.LegendSelectionEvent;
import com.sencha.gxt.chart.client.chart.event.LegendSelectionEvent.HasLegendSelectionHandlers;
import com.sencha.gxt.chart.client.chart.event.LegendSelectionEvent.LegendSelectionHandler;
import com.sencha.gxt.chart.client.chart.series.Series;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.sprite.RectangleSprite;
import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;
import com.sencha.gxt.core.shared.event.GroupingHandlerRegistration;
import com.sencha.gxt.widget.core.client.tips.ToolTip;

/**
 * Defines a legend for a chart's series. The legend class displays a list of
 * legend items each of them related with a series being rendered.
 * 
 * @param <M> the data type of the legend
 */
public class Legend<M> implements HasLegendHandlers, HasLegendSelectionHandlers, HasLegendItemOutHandlers,
    HasLegendItemOverHandlers {

  protected ToolTip toolTip;
  protected LegendToolTipConfig<M> toolTipConfig;
  private Position position = Position.BOTTOM;
  private boolean visible = true;
  private boolean created = false;
  private double padding = 5;
  private double itemSpacing = 10;
  private double width = 0;
  private double height = 0;
  private double x;
  private double y;
  private double origX;
  private double origY;
  private Chart<M> chart;
  private List<String> currentLegendTitles;
  private ArrayList<LegendItem<M>> items = new ArrayList<LegendItem<M>>();
  private RectangleSprite box;
  private Color boxStroke = RGB.BLACK;
  private double boxStrokeWidth = 1;
  private Color boxFill = RGB.WHITE;
  private HandlerManager handlerManager;
  private boolean itemHighlighting = false;
  private boolean itemHiding = false;

  @Override
  public HandlerRegistration addLegendHandler(LegendHandler handler) {
    GroupingHandlerRegistration reg = new GroupingHandlerRegistration();
    reg.add(ensureHandlers().addHandler(LegendSelectionEvent.getType(), handler));
    reg.add(ensureHandlers().addHandler(LegendItemOutEvent.getType(), handler));
    reg.add(ensureHandlers().addHandler(LegendItemOverEvent.getType(), handler));
    chart.sinkBrowserEvents();
    return reg;
  }

  @Override
  public HandlerRegistration addLegendItemOutHandler(LegendItemOutHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(LegendItemOutEvent.getType(), handler);
  }

  @Override
  public HandlerRegistration addLegendItemOverHandler(LegendItemOverHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(LegendItemOverEvent.getType(), handler);
  }

  @Override
  public HandlerRegistration addLegendSelectionHandler(LegendSelectionHandler handler) {
    chart.sinkBrowserEvents();
    return ensureHandlers().addHandler(LegendSelectionEvent.getType(), handler);
  }

  /**
   * Create all the sprites for the legend.
   */
  public void create() {
    if (box == null) {
      box = new RectangleSprite();
      box.setStroke(boxStroke);
      box.setStrokeWidth(boxStrokeWidth);
      box.setFill(boxFill);
      chart.addSprite(box);
      box.redraw();
    }
    createItems();
    if (!created && isDisplayed()) {
      created = true;
    }
  }

  /**
   * Removes all the sprites of the legend from the surface.
   */
  public void delete() {
    while (items.size() > 0) {
      items.remove(items.size() - 1).clear();
    }
    if (box != null) {
      box.remove();
      box = null;
    }
  }

  /**
   * Calculates and returns the bounding box of the legend.
   * 
   * @return the bounding box of the legend.
   */
  public PreciseRectangle getBBox() {
    return new PreciseRectangle(Math.round(x) - boxStrokeWidth / 2.0, Math.round(y) - boxStrokeWidth / 2.0, width,
        height);
  }

  /**
   * Returns the chart that the legend is attached.
   * 
   * @return the chart that the legend is attached
   */
  public Chart<M> getChart() {
    return chart;
  }

  /**
   * Returns the current list of titles used in the legend.
   * 
   * @return the current list of titles used in the legend
   */
  public List<String> getCurrentLegendTitles() {
    return currentLegendTitles;
  }

  /**
   * Returns the {@link LegendItem} at the given point. Returns null if no item
   * at that point.
   * 
   * @param point the point of the item
   * @return the legend item
   */
  public LegendItem<M> getItemFromPoint(PrecisePoint point) {
    for (int i = 0; i < items.size(); i++) {
      LegendItem<M> item = items.get(i);
      if (item.getBBox().contains(point)) {
        return item;
      }
    }
    return null;
  }

  /**
   * Returns the position of the legend.
   * 
   * @return the position of the legend
   */
  public Position getPosition() {
    return position;
  }

  /**
   * Returns the currently generated tooltip for legend.
   * 
   * @return the currently generated tooltip for legend
   */
  public ToolTip getToolTip() {
    return toolTip;
  }

  /**
   * Returns the tool tip configuration of the legend.
   * 
   * @return the tool tip configuration of the legend
   */
  public LegendToolTipConfig<M> getToolTipConfig() {
    return toolTipConfig;
  }

  /**
   * Returns true if legend item hiding is enabled.
   * 
   * @return true if legend item hiding is enabled
   */
  public boolean isItemHiding() {
    return itemHiding;
  }

  /**
   * Returns true if legend item highlighting is enabled.
   * 
   * @return true if legend item highlighting is enabled
   */
  public boolean isItemHighlighting() {
    return itemHighlighting;
  }

  /**
   * Removes the components tooltip (if one exists).
   */
  public void removeToolTip() {
    if (toolTip != null) {
      toolTip.initTarget(null);
      toolTip = null;
      toolTipConfig = null;
    }
  }

  /**
   * Sets the chart that the legend is attached.
   * 
   * @param chart the chart that the legend is attached
   */
  public void setChart(Chart<M> chart) {
    this.chart = chart;
    if (chart != null && itemHighlighting || toolTipConfig != null) {
      chart.sinkBrowserEvents();
    }
  }

  /**
   * Sets whether or not the legend uses item highlighting.
   * 
   * @param itemHiding true if the legend uses item highlighting
   */
  public void setItemHiding(boolean itemHiding) {
    this.itemHiding = itemHiding;
  }

  /**
   * Sets whether or not the legend uses item highlighting.
   * 
   * @param itemHighlighting ture if the legend uses item highlighting
   */
  public void setItemHighlighting(boolean itemHighlighting) {
    this.itemHighlighting = itemHighlighting;
    if (itemHighlighting && chart != null) {
      chart.sinkBrowserEvents();
    }
  }

  /**
   * Sets the position of the legend.
   * 
   * @param position the position of the legend
   */
  public void setPosition(Position position) {
    this.position = position;
  }

  /**
   * Sets the tooltip configuration.
   * 
   * @param config the tooltip configuration
   */
  public void setToolTipConfig(LegendToolTipConfig<M> config) {
    this.toolTipConfig = config;
    if (config != null) {
      if (chart != null) {
        chart.sinkBrowserEvents();
      }
      if (toolTip == null) {
        toolTip = new ToolTip(null, config);
      } else {
        toolTip.update(config);
      }
    } else if (config == null) {
      removeToolTip();
    }
  }

  /**
   * Adjusts the position of the legend to fit in its chart.
   */
  public void updatePosition() {
    double inset = chart.getDefaultInsets();
    PreciseRectangle bbox = chart.getBBox();
    double chartX = bbox.getX() + inset;
    double chartY = bbox.getY() + inset;
    double chartWidth = bbox.getWidth() - (inset * 2.0);
    double chartHeight = bbox.getHeight() - (inset * 2.0);
    if (isDisplayed()) {
      // Find the position based on the dimensions
      if (position == Position.LEFT) {
        x = inset;
        y = Math.floor(chartY + chartHeight / 2.0 - height / 2.0);
      } else if (position == Position.RIGHT) {
        x = Math.floor(chart.getSurface().getWidth() - width) - inset;
        y = Math.floor(chartY + chartHeight / 2.0 - height / 2.0);
      } else if (position == Position.TOP) {
        x = Math.floor(chartX + chartWidth / 2.0 - width / 2.0);
        y = inset;
      } else if (position == Position.BOTTOM) {
        x = Math.floor(chartX + chartWidth / 2.0 - width / 2.0);
        y = Math.floor(chart.getSurface().getHeight() - height) - inset;
      } else {
        x = Math.floor(origX) + inset;
        y = Math.floor(origY) + inset;
      }

      // Update the position of each item
      for (int i = 0; i < items.size(); i++) {
        items.get(i).updatePosition(x, y);
      }
      // Update the position of the containing box
      PreciseRectangle rect = getBBox();
      box.setX(rect.getX());
      box.setY(rect.getY());
      box.setWidth(rect.getWidth());
      box.setHeight(rect.getHeight());
      box.redraw();
    }
  }

  protected HandlerManager ensureHandlers() {
    if (handlerManager == null) {
      handlerManager = new HandlerManager(this);
    }
    return handlerManager;
  }

  protected void onMouseDown(LegendItem<M> item) {
    item.onMouseDown();
    ensureHandlers().fireEvent(new LegendSelectionEvent(item));
  }

  protected void onMouseMove(LegendItem<M> item) {
    item.onMouseMove();
    ensureHandlers().fireEvent(new LegendItemOverEvent(item));
  }

  protected void onMouseOut(LegendItem<M> item) {
    item.onMouseOut();
    ensureHandlers().fireEvent(new LegendItemOutEvent(item));
  }

  /**
   * Create the series markers and labels.
   */
  private void createItems() {
    double spacing = 0;
    boolean vertical = isVertical();
    double totalHeight = 0;
    double totalWidth = 0;
    double maxHeight = 0;
    double maxWidth = 0;
    double spacingOffset = 2;

    // remove all legend items
    while (items.size() > 0) {
      items.remove(0).clear();
    }

    // Create all the item labels, collecting their dimensions and positioning
    // each one
    // properly in relation to the previous item
    for (int i = 0; i < chart.getSeries().size(); i++) {
      Series<M> series = chart.getSeries(i);
      if (series.isShownInLegend()) {
        currentLegendTitles = series.getLegendTitles();
        for (int j = 0; j < currentLegendTitles.size(); j++) {
          LegendItem<M> item = new LegendItem<M>(this, series, j);
          items.add(item);
          PreciseRectangle bbox = item.getBBox();

          // always measure from x=0, since not all markers go all the way to
          // the left
          if (i + j == 0) {
            if (vertical) {
              spacing = padding + bbox.getHeight() / 2.0;
            } else {
              spacing = padding;
            }
          } else {
            if (vertical) {
              spacing = itemSpacing / 2.0;
            } else {
              spacing = itemSpacing;
            }
          }
          // Set the item's position relative to the legend box
          if (vertical) {
            item.setX(Math.floor(padding));
            item.setY(Math.floor(totalHeight + spacing));
          } else {
            item.setX(Math.floor(totalWidth + padding));
            item.setY(Math.floor(padding + bbox.getHeight() / 2.0));
          }

          // Collect cumulative dimensions
          totalWidth += bbox.getWidth() + spacing;
          totalHeight += bbox.getHeight() + spacing;
          maxWidth = Math.max(maxWidth, bbox.getWidth());
          maxHeight = Math.max(maxHeight, bbox.getHeight());
        }
      }
    }

    // Store the collected dimensions for later
    if (vertical) {
      width = Math.floor(maxWidth + padding * 2.0);
      if (items.size() == 1) {
        spacingOffset = 1;
      }
      height = Math.floor((totalHeight - spacingOffset * spacing) + (padding * 2.0));
    } else {
      width = Math.floor(totalWidth + padding * 2.0);
      height = Math.floor(maxHeight + padding * 2.0);
    }
  }

  /**
   * Returns whether or not the legend is diplayed.
   * 
   * @return true if the legend is diplayed
   */
  private boolean isDisplayed() {
    if (visible) {
      for (Series<M> series : chart.getSeries()) {
        if (series.isShownInLegend()) {
          return true;
        }
      }
      return false;
    } else {
      return false;
    }
  }

  /**
   * Returns whether or not the legend is vertical.
   * 
   * @return true if the legend is vertical
   */
  private boolean isVertical() {
    return position == Position.LEFT || position == Position.RIGHT || position == Position.DETACHED;

  }

}
