/*
 * (c) Copyright 2008, 2009 Hewlett-Packard Development Company, LP
 * All rights reserved.
 * [See end of file]
 */

package com.hp.hpl.jena.sparql.sse.writers;

import com.hp.hpl.jena.sparql.core.Prologue;
import com.hp.hpl.jena.sparql.core.TriplePath;
import com.hp.hpl.jena.sparql.path.*;
import com.hp.hpl.jena.sparql.serializer.SerializationContext;
import com.hp.hpl.jena.sparql.sse.Tags;
import com.hp.hpl.jena.sparql.util.FmtUtils;
import com.hp.hpl.jena.sparql.util.IndentedLineBuffer;
import com.hp.hpl.jena.sparql.util.IndentedWriter;

/** SSE Writer */
public class WriterPath
{
    private static final int NL = WriterLib.NL ;
    private static final int NoNL = WriterLib.NoNL ;
    private static final int NoSP = WriterLib.NoSP ;
    
    public static void write(Path path, Prologue prologue)
    {
        output(IndentedWriter.stdout, path, new SerializationContext(prologue)) ;
    }
    
    public static void output(IndentedWriter out, Path path, SerializationContext naming)
    {
        WriterPathVisitor w = new WriterPathVisitor(out, naming.getPrologue()) ;
        w.output(path) ;
        w.out.flush();
    }
    
    public static void output(IndentedWriter out, TriplePath tp, SerializationContext naming)
    {
        WriterLib.start(out, Tags.tagTriplePath, NoNL) ;
        outputPlain(out, tp, naming) ;
        WriterLib.finish(out, Tags.tagTriplePath) ;
    }
    
    public static void outputPlain(IndentedWriter out, TriplePath tp, SerializationContext naming)
    {
        boolean oneLiner = oneLiner(tp.getPath()) ;
        if ( oneLiner )
        {
            WriterNode.output(out, tp.getSubject(), naming) ;
            out.print(" ") ;
            WriterPath.output(out, tp.getPath(), naming) ;
            out.print(" ") ;
            WriterNode.output(out, tp.getObject(), naming) ;
        }
        else
        {
            nl(out, false);
            WriterNode.output(out, tp.getSubject(), naming) ;
            nl(out);
            WriterPath.output(out, tp.getPath(), naming) ;
            nl(out);
            WriterNode.output(out, tp.getObject(), naming) ;
        }
    }
    
    private static boolean oneLiner(Path path)
    {
        return (path instanceof P_Link) ;
    }
    
    private static final boolean multiline = false ;
    private static void nl(IndentedWriter out)
    {
        if ( multiline )
            out.println();
        else
            out.print(" ") ;
    }

    private static void nl(IndentedWriter out, boolean spaceForNL)
    {
        if ( multiline )
            out.println();
        else
            if ( spaceForNL ) out.print(" ") ;
    }
    
    public static String asString(Path path) { return asString(path, null) ; }
    
    public static String asString(Path path, Prologue prologue)
    {
        IndentedLineBuffer buff = new IndentedLineBuffer() ;
        WriterPathVisitor w = new WriterPathVisitor(buff.getIndentedWriter(), prologue) ;
        path.visit(w) ;
        w.out.flush();
        return buff.asString() ;
    }
    
    private static class WriterPathVisitor implements PathVisitor
    {

        private IndentedWriter out ;
        private Prologue prologue ;

        WriterPathVisitor(IndentedWriter indentedWriter, Prologue prologue)
        { 
            this.out = indentedWriter ; 
            this.prologue = prologue ;
        }

        private void output(Path path)
        {
            path.visit(this) ;
        }

        //@Override
        public void visit(P_Link pathNode)
        {
            out.print(FmtUtils.stringForNode(pathNode.getNode(), prologue)) ;
        }

        //@Override
        public void visit(P_Alt pathAlt)
        {
            visit2(pathAlt, Tags.tagPathAlt) ;
        }

        //@Override
        public void visit(P_Seq pathSeq)
        {
            visit2(pathSeq, Tags.tagPathSeq) ;
        }

        private void visit2(P_Path2 path2, String nodeName)
        {
            out.print("(") ;
            out.print(nodeName) ;
            nl(out) ; 
            out.incIndent() ;
            output(path2.getLeft()) ;
            nl(out) ; 
            output(path2.getRight()) ;
            out.decIndent() ;
            out.print(" )") ;
        }

        //@Override
        public void visit(P_Mod pathMod)
        {
            out.print("(") ;
            out.print(Tags.tagPathMod) ;
            out.print(" ") ;
            out.print(modInt(pathMod.getMin())) ;
            out.print(" ") ;
            out.print(modInt(pathMod.getMax())) ;

            if ( oneLiner(pathMod.getSubPath()) )
                out.print(" ") ;
            else
                nl(out) ;
            out.incIndent() ;
            output(pathMod.getSubPath()) ;
            out.decIndent() ;
            out.print(")") ;
        }

        private static String modInt(long value)
        {
            if ( value == P_Mod.INF ) return "*" ;
            //if ( value == P_Mod.UNSET ) return " ;

            return Long.toString(value) ;
        }

        public void visit(P_Reverse reversePath)
        {
            out.print("(") ;
            out.print(Tags.tagPathReverse) ;
            nl(out) ; 
            out.incIndent() ;
            output(reversePath.getSubPath()) ;
            out.decIndent() ;
            out.print(" )") ;
        }

    }
}

/*
 * (c) Copyright 2008, 2009 Hewlett-Packard Development Company, LP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */